(ns open-scad.models.extruder-adapter
  (:require [open-scad.core :refer :all]
            [threading.core :refer :all]))

(def mold-offset              0.4) ;; 0.5

(def screw-hole-diameter      3)
(def screw-head-diameter      5.5)
(def screw-head-height        3)
(def adapter-height           (+ screw-head-height 1))
(def adapter-diameter         36)
(def motor-holes-distance     25.5)
(def motor-hole-diameter      3)
(def extruder-holes-distance  26.5)
(def extruder-hole-diameter   3.5)
(def center-hole-diameter     23)

(def guide-shaft-height       5)
(def guide-outer-diameter     5)
(def guide-inner-diameter     2)
(def guide-hat-diameter       (+ 2 guide-outer-diameter))
(def guide-hat-height         3)
(def guide-tip-outer-diameter 6)
(def guide-tip-inner-diameter 4)
(def guide-tip-height         2)
(def guide-tip-width          0)

(def total-height             (+ guide-shaft-height
                                 guide-hat-height
                                 guide-tip-height))

(defgeometry center-hole []
  (->> (cylinder (/ center-hole-diameter 2) (+ 2 adapter-height) :center false)
       (translate [0 0 -1])))

(defgeometry holes [dist & {:keys [head]}]
  (let [d    (/ (+ dist screw-hole-diameter) 2)
        hole (->> (cylinder (/ screw-hole-diameter 2) (+ 2 adapter-height)
                            :center true)
                  (translate [0 0 (inc (/ adapter-height 2))]))
        h    (->> (cylinder (/ screw-head-diameter 2) (+ 1 screw-head-height)
                            :center false)
                  (translate [0 0 (inc (- adapter-height screw-head-height))]))]
    ; (println "d" d)
    ; (println "ref-diameter" ref-diameter)
    ; (translate [d 0 0] hole)
    (•- (-> hole (when-> (<- head) (union h)))
        (->> (<<- (union (-•• (translate [0 (+ d) 0]))
                         (-•• (translate [0 (- d) 0]))
                         (-•• (translate [(+ d) 0 0]))
                         (-•• (translate [(- d) 0 0]))))
             (translate [0 0 -1])))))

(defgeometry extruder-adapter []
  (-> (cylinder (/ adapter-diameter 2) adapter-height :center false)
      (difference (center-hole)
                  (holes motor-holes-distance :head true)
                  (->> (holes extruder-holes-distance)
                       (rotate [0 0 (° 45)])))))



(defgeometry filament-guide []
  (let [id            guide-tip-inner-diameter
        od            guide-tip-outer-diameter
        dig-shaft     (->> (cylinder (+ (/ guide-inner-diameter 2) 0.4)
                                     (+ 2 guide-shaft-height))
                           (translate [0 0 2]))
        central-shaft (difference
                        (cylinder (- (/ guide-outer-diameter 2) mold-offset)
                                  guide-shaft-height :center false)
                        dig-shaft)
        cyl           (cylinder (/ guide-hat-diameter 2)
                                  guide-hat-height :center false)
        hat           (->> (difference cyl dig-shaft)
                           (translate [0 0 guide-shaft-height]))
        tip-circle    (->> (cylinder (/ od 2) guide-hat-diameter)
                           (rotate [0 (° 90) (° 90)]))
        dig-circle    (->> (cylinder (/ id 2) (+ 2 guide-hat-diameter))
                           (rotate [0 (° 90) (° 90)]))
        tip-thickness (- guide-outer-diameter guide-inner-diameter)
        dig-tip       (cube (- guide-outer-diameter tip-thickness)
                            (- guide-outer-diameter tip-thickness)
                            (- guide-tip-height     tip-thickness))
        box           (cube guide-hat-diameter guide-hat-diameter
                            guide-tip-height)
        tip-box       (->> (difference box dig-tip)
                           (translate [0 0 (/ guide-tip-height -2)]))
        x             (/ (- od (/ (- od id) 2)) 2)
        tip           (->
                        tip-box
                        (union (translate [(- x) 0 0] tip-circle)
                               (translate [(+ x) 0 0] tip-circle))
                        (difference (translate [(- x) 0 0] dig-circle)
                                    (translate [(+ x) 0 0] dig-circle)
                                    (translate [0 0 (/ guide-tip-height 2)]
                                               (scale [1.1 1.1 1] box))
                                    (translate [guide-outer-diameter 0 0]
                                               (scale [1 1.1 2] box))
                                    (translate [(- guide-outer-diameter) 0 0]
                                               (scale [1 1.1 2] box)))
                        (intersection (->> cyl
                                    (translate [0 0 (- guide-hat-height)])))
                        (->> (translate
                               [0 0 (+ guide-shaft-height guide-hat-height
                                       guide-tip-height)])))]
    (union central-shaft
           (-> (union hat tip)
               (difference (translate [0 0 (dec (/ guide-shaft-height 2))]
                                      (scale [1 1 2] dig-shaft)))))))



(render ($fn 40 (union (part :adapter (extruder-adapter))
                       (->> (part :guide (filament-guide))
                            (translate [-30 0 0])))))
