(ns open-scad.models.hydraulic-adapters
  (:use clojure.pprint)
  (:require [open-scad.core                        :refer :all]
            [scad-clj.model                        :as    m]
            [open-scad.libs.threads                :refer [metric-thread]]
            [open-scad.libs.threads2               :refer [BSP-thread]]
            [open-scad.libs.universal-hose-adapter :refer [hose]]
            [open-scad.libs.threadlib              :refer [thread bolt nut tap]]
            [open-scad.libs.nosecones              :refer [cone-elliptical]]
            [open-scad.libs.tsmthread4             :refer [thread-npt]]))


(def good-pump-adapter-thickness  1.5)
(def good-pump-screw-diameter     19)
(def good-pump-screw-height       15.5)
(def good-pump-screw-pitch        (* 10 (/ (inch 1) 16)))
(def good-pump-cone-height        (+ 7.3 5))
(def good-pump-cone-top-diameter  9)
(def good-pump-top-shaft-diameter 5)
(def good-pump-top-shaft-height   7)
(def good-pump-top-screw-height   6)
(def good-pump-top-cap-height     12)
(def good-pump-top-cap-depth      9.5)
(def good-pump-cap-top-thickness  (- good-pump-top-cap-height
                                     good-pump-top-cap-depth))
(def good-pump-cap-hole-diameter  6)
(def good-pump-cap-diameter       15.75)

(defgeometry hollow-cylinder [inner-d height thickness]
  (difference (cylinder (+ (/ inner-d 2) thickness) height :center false)
              (->> (cylinder (/ inner-d 2) (+ 2 height) :center false)
                   (translate [0 0 -1]))))

(defn bsp [ratio length & {:keys [internal]}]
  ; (BSP-thread
  ;   :nominal-pipe-size 1/8
  ;   :length            length
  ;   :internal          internal)

  ; (let [diameter (case ratio
  ;                  1/8 0.383
  ;                  1/4 0.518)
  ;       pitch    (case ratio
  ;                  1/8 28
  ;                  1/4 19)]
  ;   (metric-thread
  ;     (* 25.4 diameter)
  ;     (* 25.4 (/ 1 pitch))
  ;     length
  ;     :internal internal
  ;     :leadin 1))

  (let [bolt-designator (str "G" ratio)
        designator (str bolt-designator (if internal "-int" "-ext"))]
    (bolt bolt-designator
          :turns (literal
                   (format "%s / thread_specs(\"%s\", table=THREAD_TABLE)[0]"
                           (str (float length)) designator)))))

(defn npt [ratio length & {:keys [internal]}]
  (let [diameter (case ratio
                   1/8 0.405
                   1/4 0.540)
        pitch    (case ratio
                   1/8 27
                   1/4 18)]
    (metric-thread
      (* 25.4 diameter)
      (* 25.4 (/ 1 pitch))
      length
      :internal internal)))

(defgeometry good-pump-adapter []
  (let [pump-side-thread (difference
                           (cylinder (+ (* 2 good-pump-adapter-thickness)
                                        (/ good-pump-screw-diameter 2))
                                     good-pump-screw-height :center false)
                           (->> (metric-thread
                                  good-pump-screw-diameter
                                  good-pump-screw-pitch
                                  (+ 2 good-pump-screw-height)
                                  :internal true
                                  :thread-size 1)
                                (translate [0 0 -1])))
        middle           (hose (+ good-pump-screw-diameter
                                  (* 2 2 good-pump-adapter-thickness))
                               (+ (+ good-pump-top-shaft-diameter
                                     (* 2 good-pump-adapter-thickness))
                                  good-pump-adapter-thickness)
                               good-pump-cone-height
                               (* 2 good-pump-adapter-thickness)
                               0 0 0 0)
        top              (difference
                           (bsp 1/8 good-pump-top-screw-height)
                           (->> (cylinder (- (/ good-pump-top-shaft-diameter 2)
                                             (/ good-pump-adapter-thickness 2))
                                          (+ good-pump-top-screw-height 2))
                                (translate
                                  [0 0 (/ good-pump-top-screw-height 2)])))
        cherry           (hose good-pump-top-shaft-diameter
                               good-pump-top-shaft-diameter
                               good-pump-top-shaft-height
                               (/ good-pump-adapter-thickness 2)
                               1 4 5 1)]
    (union #_pump-side-thread
           (->> middle
                (translate [0 0 good-pump-screw-height]))
           (->> top
                (translate [0 0 (+ good-pump-screw-height
                                   good-pump-cone-height)]))
           (->> cherry
                  (translate [0 0 (+ good-pump-screw-height
                                     good-pump-cone-height
                                     good-pump-top-screw-height)])))))

(defgeometry good-pump-adapter-cap []
  (difference (cylinder (/ good-pump-cap-diameter 2)
                        good-pump-top-cap-height
                        :center false)
              (->> (cylinder (/ good-pump-cap-hole-diameter 2)
                             (+ 2 good-pump-top-cap-height)
                             :center false)
                   (translate [0 0 -1]))
              (->> (bsp 1/8
                        (/ (+ 1 good-pump-top-cap-depth) 25.4)
                        :internal true)
                   (translate [0 0 -1]))))

(defgeometry ^{:foreign-name cylinder} cyl
  [& {:keys [h r r1 r2 d d1 d2 center $fa $fs $fn]}])

(def holder-diameter        15)
(def female-thread-height   10)
(def female-thread-diameter (+ 10 1))
(def female-thread-pitch    1)
(def male-thread-height     10)
(def male-thread-ratio      1/8)
(def filter-height          (+ 15 2))
(def central-pipe-height    (+ female-thread-height filter-height))
(def wall-width             2)
(def male-inner-diameter    4)

(defgeometry nozzle-holder [& {:keys [shape]}]
  (let [body  (-% (cyl :d holder-diameter
                       :h (+ central-pipe-height wall-width)
                       :center false
                       :$fn 6))
        thf   (->> ; #_(metric-thread female-thread-diameter
                   ;                  female-thread-pitch
                   ;                  (+ 1 female-thread-height)
                   ;                  :internal true)

                   (thread-npt {:DMAJ (literal (+ 1 10.287))
                                :PITCH 0.941
                                :L (+ 1 female-thread-height)})
                   (translate [0 0 -1]))
        hol1  (->> (cone-elliptical {:R (/ female-thread-diameter 2)
                                     :L (+ filter-height 1)
                                     :s (literal "$fn")})
                   (translate [0 0 (- female-thread-height 1)]))
        thm   (bsp male-thread-ratio male-thread-height
                   :internal false)
        hol2  (->> (cyl :d male-inner-diameter
                        :h (+ wall-width male-thread-height 4)
                        :center false)
                   (translate [0 0 (- central-pipe-height 2)]))]
    ; (union (translate [0 0 1.5] (cube 10 10 3))
    ;        (translate [0 0 3] thm))
    (case shape
      :I (-> (difference (union body (->> thm
                                          (translate [0 0 (+ central-pipe-height
                                                             wall-width)])))
                         (union hol1 hol2 thf
                                #_(translate [0 0 -1] (cyl :d 10.4 :h 11)))))
      :T (let [bod          (difference body (union hol1 hol2 thf))
               f             (/ 1 6 2)
               center-length 15
               new-hold-diam (* holder-diameter
                                (Math/sin (- (/ π 2)
                                             (* 2 π f))))
               place         #(->> %
                                   (rotate [0 0 (° (* 360 f))])
                                   (rotate [0 (° 90) 0])
                                   (translate [0 0 (/ new-hold-diam 2)]))
               branch        (->> bod
                                  place
                                  (translate [(- (+ central-pipe-height
                                                    wall-width
                                                    (/ center-length 2)))
                                              0
                                              0]))

               center        (->> (cyl :d holder-diameter
                                       :h center-length
                                       :center false
                                       :$fn 6)
                                  place
                                  (translate [(- (/ center-length 2)) 0 0]))
               hol-horiz-l   (+ (* 2 wall-width) center-length 10)
               hol-horiz     (->> (cyl :d male-inner-diameter
                                       :h hol-horiz-l
                                       :center false)
                                  (rotate [0 (° 90) 0])
                                  (translate [(- (/ hol-horiz-l 2))
                                              0
                                              (/ new-hold-diam 2)]))
               hol-vert      (->> (cyl :d male-inner-diameter
                                       :h (+ (/ new-hold-diam 2)
                                             wall-width
                                             male-thread-height
                                             1))
                                  (translate [0 0 (/ new-hold-diam 2)]))
               vert          (union (translate [0 0 new-hold-diam] thm)
                                    (translate [0 0 (/ new-hold-diam 2)]
                                               (cyl :d holder-diameter
                                                    :h (/ new-hold-diam 2)
                                                    :center false
                                                    :$fn 6)))]
           (-> (union branch
                      (mirror [1 0 0] branch)
                      center
                      vert)
               (difference (union hol-horiz hol-vert)))))))

; (render ($fn 1000 [#_(part :gp-adapter     (good-pump-adapter))
;                  #_(->> (part :gp-adapter-cap (good-pump-adapter-cap))
;                         (translate [0 0 39]))
;                  (part :nozzle-holder (nozzle-holder))]))

(render ($fn 300 [#_(part :nozzle-holder   (nozzle-holder :shape :I))
                  (part :nozzle-holder-T (nozzle-holder :shape :T))]))
