(ns open-scad.models.hydroponic-tube-holder
  (:require [open-scad.core :refer :all]
            [threading.core :refer :all]))

(def tube-diameter            140)
(def holder-height            (/ tube-diameter 4))
(def holder-z-offset          20)
(def holder-mask-z-offset     (/ tube-diameter 2))
(def plank-thickness          (+ 16 #_14.7 0.3))
(def leg-length               40)
(def leg-width                20)
(def leg-thickness            5)
(def holder-thickness         (+ plank-thickness (* 2 leg-thickness)))
(def foot-thickness           8)

(def leg-screw-diameter       5.8)
(def leg-screw-hole-diameter (+ leg-screw-diameter 2))
(def leg-head-diameter       (+ 11.15 0.5))
(def leg-head-notch-depth    2)

(defgeometry leg [& {:keys [bolt-notch]}]
  (let [hole       (cylinder (/ leg-screw-hole-diameter 2) (+ 1 leg-thickness))
        head-notch (cylinder (/ (+ 1 leg-head-diameter) 2) leg-thickness)]
    (difference (cube leg-length leg-width leg-thickness)
                hole
                (translate [0 0 (- leg-head-notch-depth leg-thickness)]
                           (-> head-notch
                               (when-> (<- bolt-notch) (->> ($fn 6)))))
                (when bolt-notch
                  ))))

(def notch-thickness     2.5)
(def notch-r-inset       5)
(def notch-angle         5)
(def notch-opening-angle 2.5)
(def notch-rotate-angle  120)

(defgeometry elastic-notch [r]
  (->> (square notch-thickness holder-thickness)
       (translate [r 0 0])
       (extrude-rotate {:angle notch-angle})
       (union (->> (square (+ notch-thickness notch-r-inset)
                           holder-thickness)
                   (translate [(+ r notch-thickness) 0 0])
                   (extrude-rotate {:angle notch-opening-angle})
                   (rotate [0 0
                            (° (- notch-angle notch-opening-angle))])))
       (scale [1 1 1.1])))



(def attach-angle     64)
(def attach-thickness 10)

(defgeometry elastic-attach []
  (let [notch (->> (elastic-notch (- (/ (+ tube-diameter
                                           attach-thickness)
                                        2)
                                     notch-r-inset))
                   (rotate [0 0 (° (* 2 (- notch-angle)))]))]
    (->> (square attach-thickness holder-thickness)
         (translate [(/ tube-diameter 2) 0 0])
         (extrude-rotate {:angle (/ attach-angle 2)})
         (rotate [0 0 (° (- (/ attach-angle 4)))])
         (>>- (-> (difference [notch (mirror [0 1 0] notch)]))))))


(def band-length      100)
(def band-thickness   2)
(def band-hole-dist   4)
(def band-width       (+ (* 2 band-hole-dist) holder-thickness))
(def band-hole-length 8)
(def band-hole-width  holder-thickness)

; (defgeometry elastic-band []
;   (let [offsets         (->> (repeat [band-hole-dist band-hole-length])
;                              (reductions (partial apply +) 0)
;                              (take-while #(< % band-length)))
;         new-band-length (-> offsets last (+ (* 2 band-hole-dist)
;                                             band-hole-length))
;         band            (->> (cube new-band-length band-width band-thickness)
;                              (translate [(/ new-band-length 2) 0 0]))
;         hole            (->> (cube band-hole-length
;                                    band-hole-width
;                                    band-thickness)
;                              (translate [band-hole-length 0 0])
;                              (scale [1 1 1.1]))]
;     (difference band
;            (for [x offsets]
;              (translate [x 0 0]
;                         hole)))))

(defgeometry elastic-band []
  (let [r            2
        R            2
        RR           (+ r R)
        h            (* 2 r)
        snake-dist   8
        semi-ring    (->> (square r h)
                          (translate [RR 0 0])
                          (extrude-rotate {:angle 180})
                          (translate [RR 0]))
        nb-rings     (Math/floor (/ band-length RR 2))
        snake        (for [x (range 1 nb-rings)]
                       (-> semi-ring
                           (when-not-> (<- (zero? (mod x 2)))
                             (->> (mirror [0 1 0])))
                           (->> (translate [(+ (* x 2 RR) (/ r 2))
                                            (+ RR r (/ (+ RR) 2))
                                            (/ h 2)]))))
        nb-snakes    (dec (Math/floor (/ band-width snake-dist)))
        snakes       (for [x (range (inc nb-snakes))]
                       (->> snake
                            (translate [0
                                        (* x snake-dist)
                                        0])))
        l            (+ (* 2 band-thickness) (* nb-rings RR 2))
        w            (+ (* 2 band-thickness) (* nb-snakes snake-dist) (/ r 2))
        h            (* 2 r)
        attach       (->> (cube (+ band-hole-length (* 2 band-thickness))
                                (+ w (* 2 band-thickness))
                                h
                                :center false)
                          (translate [0 0 0]))
        attach-hole  (->> (cube band-hole-length band-hole-width (+ 2 h)
                                :center false)
                          (translate [band-thickness
                                      (+ (/ (- w band-hole-width) 2)
                                         band-thickness)
                                      -1]))
        place        #(->> % (translate [(- l band-hole-length) 0 0]))
        bar-ratio    1/2
        bar-f        (/ 1 bar-ratio)
        bars         (for [x (range 1 (Math/floor (/ band-length RR 2 bar-f)))]
                       (->> (cube r h (- w (/ snake-dist 2)) :center false)
                            (rotate [(° -90) 0 0])
                            (translate [(+ RR (* x 2 bar-f RR) (- (/ r 2)))
                                        (+ snake-dist (/ RR 2) r)
                                        h])))]
    snakes
    (-> (union snakes
               (->> bars (translate [1 0 0]))
               attach
               (place attach))
        (difference attach-hole (place attach-hole)))))

(defgeometry tube-holder []
  (let [h    (cylinder (/ tube-diameter 2) holder-thickness)
        w    (* 2/3 tube-diameter)
        foot (->> (cube foot-thickness w holder-thickness)
                  (translate [(- (* 1/2 (- tube-diameter foot-thickness)))
                              0 0]))]
    (-> h
        (difference (->> h (translate [holder-z-offset 0 0])
                         (scale [1 1 1.1])))
        (union foot)
        (difference (->> (cube (* 7/5 tube-diameter) tube-diameter holder-thickness)
                         (translate [holder-mask-z-offset 0 0])
                         (scale [1 1 1.1])))
        (union
          (for [[rota? y-dir] [[false  +] [false -]]
                [rotb? z-dir] [[false  +] [true  -]]]
            (->> (leg :bolt-notch (when rotb? true))
                 (>>- (when-> (<- (not rotb?))
                        (->> (rotate [(° 180) 0 0]))))
                 (translate
                   [(- (* 1/2 (+ leg-length tube-diameter)))
                    (y-dir (* 1/2 (- w leg-width)))
                    (z-dir (* 1/2 (- holder-thickness leg-thickness)))]))))
        (difference (let [h (->> (elastic-notch
                                   (- (/ tube-diameter 2)
                                      notch-r-inset))
                                 (rotate
                                   [0 0 (° notch-rotate-angle)]))]
                      [h (mirror [0 1 0] h)])))))


(defgeometry leg-distance-test [holder]
  (let [c (cube (- tube-diameter (* 2 foot-thickness))
                tube-diameter
                (* 2 holder-thickness))]
    (difference holder
           c
           (translate [(- (+ leg-length foot-thickness 1)) (- leg-width) 0] c))))

(defgeometry leg-hole-test [holder]
  (difference (leg-distance-test holder)
              (->> (cube (- tube-diameter (* 2 foot-thickness))
                         tube-diameter
                         (* 2 holder-thickness))
                   (translate [(- (/ tube-diameter 2))
                               1
                               (+ (/ holder-thickness 2)
                                  leg-thickness)]))))



(render ($fn 30
             #_(part :leg-distance-test (leg-distance-test (tube-holder)))
             #_(part :leg-hole-test (leg-hole-test (tube-holder)))
             [(part :tube-holder (tube-holder))
              (part :elastic-attach (elastic-attach))
              (->> (part :elastic-band (elastic-band))
                   (translate [0 (+ 40 (/ tube-diameter 2)) 0]
                              (elastic-band)))]))
