(ns clinicaltrials-gov.core
  (:require [clj-http.client :as client]
            [clojure.xml :as xml]
            [clojure.zip :as zip])
  (:gen-class)
  (:import (java.io InputStream)
           (java.util.zip ZipFile)))

(def api-base "https://clinicaltrials.gov/search")

(defn ^:private temp-file
  []
  (java.io.File/createTempFile "clinical-trials-gov-" ".zip"))

(defn ^:private process-status
  "Takes :open, :exclude-unknown, or a vector of both and returns a map of the
  parameters per the clinicaltrials.gov API"
  [status]
  (case status
    :open {:recr "Open"}
    :exclude-unknown {:no_unk "Y"}
    ; else, if it's a collection, recursively call this function
    (apply merge (mapv process-status status))))

(defn ^ZipFile search*
  "Executes a search request and saves the resulting ZIP file as a tmp file,
  returning a handle to the file

  Input map:

  :terms - search terms as a string (passed straight through to API)
  :status - :open, :exclude-unknown, or vector of both"
  [^String terms, status]
  (let [status-params (process-status status)
        params (merge status-params {:studyxml "true"
                                     :term terms})
        output-file (temp-file)
        response (client/get api-base {:query-params params, :as :stream, :debug false})]
    (clojure.java.io/copy
      (:body response)
      output-file)
    (ZipFile. output-file)))

(defn ^:private entries
  "Convenience wrapper around java api"
  [zip-file]
  (enumeration-seq (.entries zip-file)))

(defn ^:private ^InputStream input-stream
  "Convenience wrapper around java api"
  [zip-file entry]
  (.getInputStream zip-file entry))

(defn ^:private unzipfile-results
  "Takes a file handle (as returned by search*) and unzips (decompresses) and
  slurps the XML files, returning a vector of xml documents as strings"
  [file]
  (with-open [z file]
    (mapv #(slurp (input-stream z %1)) (entries z))))

(defn xml-input-streams
  "Takes a ZipFile handle (as returned by search*) and unzips (decompresses) and
  returns a vector of input streams to each XML file.  Requires external
  closing of the zip file"
  [^ZipFile z]
  (mapv #(input-stream z %1) (entries z)))

(defn zip-str [s]
  "Takes XML as a string and returns a zip (in the zipper not zipfile sense)"
  (zip/xml-zip
    (xml/parse (java.io.ByteArrayInputStream. (.getBytes s)))))

(defn ^:private results-zipper
  "Convenience function to take all results and return their zipper form"
  [results]
  (mapv zip-str results))

(defn search
  "Executes the search and returns a vector of zippable trees"
  [^String terms, status]
  (results-zipper (unzipfile-results (search* terms status))))

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (println "Hello, World!"))
