# clj-opentracing
[![Clojars Project](https://img.shields.io/clojars/v/org.zalando.automata/clj-opentracing.svg)](https://clojars.org/org.zalando.automata/clj-opentracing)


```clj
[org.zalando.automata/clj-opentracing "0.0.0"]
```

## Usage

`clj-opentracing` only relies on a generic tracing library for JVM: `com.lightstep.tracer/lightstep-tracer-jre`. However,
in order to work, it requires additional libraries, depending on the transport that your collectors are using:

Include the following libraries for GRPC transport (use latest available versions):

```edn
[com.lightstep.tracer/tracer-grpc "0.15.10"]
[io.grpc/grpc-netty "1.18.0"]
[io.netty/netty-tcnative-boringssl-static "2.0.20.Final"]
```

For HTTPS transport:

```edn
[com.lightstep.tracer/tracer-okhttp "0.16.0"]
```

### Creating a singleton tracer

The following example uses [mount-lite]:

```clj
(ns cdp-log-service.tracing
  (:require [mount.lite :as m]
            [clj-opentracing.tracing-singleton :as tracing])
  (:import [com.lightstep.tracer.jre JRETracer]
           [com.lightstep.tracer.shared Options$OptionsBuilder]))

(m/defstate tracer
  :start (if-not LIGHTSTEP_TOKEN
           (log/warn "LIGHTSTEP_TOKEN is not set, distributed tracing is disabled.")
           (do
             (log/info "Starting Tracing for component %s on %s" LIGHTSTEP_COMPONENT_NAME LIGHTSTEP_COLLECTOR_HOST)
             (let [res (try
                         (JRETracer. (-> (Options$OptionsBuilder.)
                                         (.withAccessToken LIGHTSTEP_TOKEN)
                                         (.withComponentName LIGHTSTEP_COMPONENT_NAME)
                                         (.withCollectorHost LIGHTSTEP_COLLECTOR_HOST)
                                         (.withCollectorPort LIGHTSTEP_COLLECTOR_PORT)
                                         (.build)))
                         (catch Exception e
                           (log/error e "Failed to create a LightStep client.")))]
               (tracing/set-tracer! res)
               res)))
  :stop (some-> @tracer (.flush 10000)))
```

`(tracing/set-tracer! res)` provides the global variable value with the current tracer.

If you don't call `set-tracer!`, all the tracing functions will just do nothing.


### Using the tracer

```clj
(ns my-project.other-namespace
  (:require [clj-opentracing.singleton-tracer :as tracing]))

;; Just some function that is traced
(defn some-function [x]
  (tracing/with-span "some-function"
    (tracing/log! {:x x})
    0))

;; Call to another service that expects tracing context in HTTP header
(defn request-impl [opts]
  (tracing/with-span "other-service/request-impl"
    (tracing/set-tags! {:span.kind     "client"
                        :component     "other-service"
                        :peer.hostname (some-> opts :url url/url :host)
                        :peer.service  "other-service"
                        :http.method   (-> opts :method name str/upper-case)
                        :http.url      (:url opts)})
    (http/request (-> {:as                   :json
                       :conn-request-timeout 10000
                       :conn-timeout         1000
                       :socket-timeout       5000}
                      (merge opts)
                      (merge {:oauth-token (credentials/get :other-service-token-secret)})
                      (update :headers merge (tracing/get-current-span-context-headers))))))

;; Extracting tracing context from incoming HTTP request headers
(defn make-handler []
  (-> (routes
        (GET "/.health" _ {:status 200})
        (route/not-found nil))
      (tracing/wrap-parent-span-context)))
```

### Using explicit tracer

Alternatively, you can include `clj-opentracing.explicit-tracer` name and give the tracer explicitly to all `tracing/*` functions:

```clj
(defn some-function [tracer x]
  (tracing/with-span tracer "some-function"
    (tracing/log! tracer {:x x})
    0))
```

All the same functions are available.

In this case you'll have to pass around the tracer object to every place where tracing is needed.


## License

Copyright © 2019 Zalando SE

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.


[mount-lite]: https://github.com/aroemers/mount-lite
