(ns clj-opentracing.explicit-tracer
  (:require [clojure.tools.logging :as log])
  (:import (io.opentracing.propagation TextMapInjectAdapter Format$Builtin TextMapExtractAdapter)
           (java.util HashMap)))


(defn- get-current-span [tracer]
  (some-> tracer .scopeManager .active .span))


(defmacro ^:private assert-inside-span [tracer & body]
  `(when ~tracer
     (let [current-span# (get-current-span ~tracer)]
       (assert current-span# "Must be called from inside a span.")
       (when current-span#
         ~@body))))


(defn set-tag! [tracer k v]
  (assert-inside-span tracer
    (.setTag (get-current-span tracer) (name k) (str v))))


(defn set-tags! [tracer tag-map]
  (doseq [[k v] tag-map]
    (set-tag! tracer k v)))


(defn normalize-string-or-map [string-or-map]
  (if (map? string-or-map)
    (into {} (for [[k v] string-or-map
                   :when (some? v)
                   :when (some? v)]
               [(name k) v]))
    (if (nil? string-or-map)
      ""
      string-or-map)))


(defn log!
  ([tracer string-or-map]
   (assert-inside-span tracer
     (.log (get-current-span tracer) (normalize-string-or-map string-or-map))))
  ([tracer timestamp string-or-map]
   (assert-inside-span tracer
     (.log (get-current-span tracer) timestamp (normalize-string-or-map string-or-map)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Creating spans


;; Normally only set from a context extracted from HTTP headers
(def ^:dynamic ^:private *parent-span-context* nil)


(defn- get-current-span-context [tracer]
  (if-let [current-span (get-current-span tracer)]
    (.context current-span)
    *parent-span-context*))


(defn with-span-impl [tracer span-name f]
  (if-not tracer
    (f)
    (with-open [_ (-> tracer
                      (.buildSpan span-name)
                      (.asChildOf (get-current-span-context tracer))
                      (.startActive true))]
      (f))))


(defmacro with-span [tracer span-name & body]
  `(with-span-impl ~tracer ~span-name (fn [] ~@body)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Using parent spans created by our server's clients


(defn with-parent-span-context-impl [tracer context f]
  (if-not tracer
    (f)
    (do
      (assert (nil? (get-current-span tracer)) "Must be called outside of any span")
      (assert (nil? *parent-span-context*) "Parent span context already set")
      (binding [*parent-span-context* context]
        (f)))))


(defmacro with-parent-span-context [tracer context & body]
  `(with-parent-span-context-impl ~tracer ~context (fn [] ~@body)))


(defn- headers->span-context [tracer headers]
  (when tracer
    (try
      (let [extract-adapter (TextMapExtractAdapter. headers)]
        (.extract tracer Format$Builtin/HTTP_HEADERS extract-adapter))
      (catch Exception e
        (log/warn e "Unable to parse parent span headers.")))))


(defn wrap-parent-span-context [handler tracer]
  (fn [request]
    (with-parent-span-context tracer (headers->span-context tracer (:headers request))
                              (handler request))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; For us as a client to include our current span context in a HTTP request to another service


(defn get-span-context-headers [tracer]
  (when tracer
    (when-let [span-context (get-current-span-context tracer)]
      (let [mutable-carrier-map (HashMap.)
            inject-adapter      (TextMapInjectAdapter. mutable-carrier-map)]
        (.inject tracer span-context Format$Builtin/HTTP_HEADERS inject-adapter)
        (into {} mutable-carrier-map)))))
