# clj-opentracing
[![Clojars Project](https://img.shields.io/clojars/v/org.zalando.automata/clj-opentracing.svg)](https://clojars.org/org.zalando.automata/clj-opentracing)


```clj
[org.zalando.automata/clj-opentracing "0.2.1"]
```

## Usage

`clj-opentracing` only relies on a generic tracing library for JVM: `com.lightstep.tracer/lightstep-tracer-jre`. However,
in order to work, it requires additional libraries, depending on the transport that your collectors are using:

Include the following libraries for GRPC transport (use latest available versions):

```edn
[com.lightstep.tracer/tracer-grpc "0.15.10"]
[io.grpc/grpc-netty "1.18.0"]
[io.netty/netty-tcnative-boringssl-static "2.0.20.Final"]
```

For HTTPS transport:

```edn
[com.lightstep.tracer/tracer-okhttp "0.16.0"]
```

### Creating a singleton tracer

The following example uses [mount-lite]:

```clj
(ns my-project.tracing
  (:require [mount.lite :as m]
            [clj-opentracing.tracing-singleton :as tracing])
  (:import [com.lightstep.tracer.jre JRETracer]
           [com.lightstep.tracer.shared Options$OptionsBuilder]))

(m/defstate tracer
  :start (if-not LIGHTSTEP_TOKEN
           (do
             (tracing/set-tracer! nil)
             (log/warn "LIGHTSTEP_TOKEN is not set, distributed tracing is disabled."))
           (do
             (log/info "Starting Tracing for component %s on %s" LIGHTSTEP_COMPONENT_NAME LIGHTSTEP_COLLECTOR_HOST)
             (let [res (try
                         (JRETracer. (-> (Options$OptionsBuilder.)
                                         (.withAccessToken LIGHTSTEP_TOKEN)
                                         (.withComponentName LIGHTSTEP_COMPONENT_NAME)
                                         (.withCollectorHost LIGHTSTEP_COLLECTOR_HOST)
                                         (.withCollectorPort LIGHTSTEP_COLLECTOR_PORT)
                                         (.build)))
                         (catch Exception e
                           (log/error e "Failed to create a LightStep client.")))]
               (tracing/set-tracer! res)
               res)))
  :stop (some-> @tracer (.flush 10000)))
```

`(tracing/set-tracer! res)` provides the global variable value with the current tracer.

If you don't call `set-tracer!`, all the tracing functions will just do nothing.

**IMPORTANT** If you are using mount or [mount-lite], you need to also require `my-project.tracing` together with
`clj-opentracing.singleton-tracer` to ensure the correct dependency graph.


### Using the tracer

Tracing arbitrary functions:

```clj
(ns my-project.other-namespace
  (:require [my-project.tracing] ;; needed to ensure mount state startup order
            [clj-opentracing.singleton-tracer :as tracing]))

;; Just some function that is traced
(defn some-function [x]
  (tracing/with-span "some-function"
    (tracing/log! {:x x})
    0))
```

Integration with HTTP client library [clj-http](https://github.com/dakrone/clj-http):

```clj
(require '[clj-http.client :as http])

;; Call to an external service that expects tracing context in HTTP headers
(defn request-impl [opts]
  (tracing/with-span "other-service/request-impl"
    (tracing/set-tags! {:span.kind     "client"
                        :component     "other-service"
                        :peer.hostname (some-> opts :url url/url :host)
                        :peer.service  "other-service"
                        :http.method   (-> opts :method name str/upper-case)
                        :http.url      (:url opts)})
    (http/request (-> {:as                   :json
                       :conn-request-timeout 10000
                       :conn-timeout         1000
                       :socket-timeout       5000}
                      (merge opts)
                      (merge {:oauth-token (credentials/get :other-service-token-secret)})
                      (update :headers merge (tracing/get-current-span-context-headers))))))
```

Integration with HTTP server (Ring-style), `wrap-span-from-request` sets tags for both request and response metadata:

```clj
(require '[compojure.core :refer :all]
         '[compojure.route :as route])

;; Extracting tracing context from incoming HTTP request headers
;; Middleware rule of thumb: Request goes bottom to top, response goes top to bottom
(defn make-handler []
  (-> (routes
        (GET "/.health" _ {:status 200})
        (route/not-found nil))
      (tracing/wrap-span "Another span" {:foo "bar})
      (tracing/wrap-span-from-request "Handle HTTP request" {:version (u/implementation-version)})))
```

Integration with [HugSQL](https://www.hugsql.org/), extending multimethods:

```clj
(defn execute-with-tracing [this db sqlvec {:keys [fn-name] :as options}]
  (tracing/with-span "hugsql-write"
    (tracing/set-tags! {:fn-name fn-name
                        :query   (first sqlvec)})
    (hugsql.adapter/execute this db sqlvec options)))


(defn query-with-tracing [this db sqlvec {:keys [fn-name] :as options}]
  (tracing/with-span "hugsql-select"
    (tracing/set-tags! {:fn-name fn-name
                        :query   (first sqlvec)})
    (hugsql.adapter/query this db sqlvec options)))


(defmethod hugsql.core/hugsql-command-fn :? [sym] `query-with-tracing)
(defmethod hugsql.core/hugsql-command-fn :! [sym] `execute-with-tracing)
```

### Using explicit tracer

Alternatively, you can use `clj-opentracing.explicit-tracer` namespace instead of `clj-opentracing.singleton-tracer`
and give the tracer explicitly to all `tracing/*` functions:

```clj
(ns my-project.other-namespace
  (:require [clj-opentracing.explicit-tracer :as tracing]))

(defn some-function [tracer x]
  (tracing/with-span tracer "some-function"
    (tracing/log! tracer {:x x})
    0))
```

All the same functions are available.

In this case you'll have to pass around the tracer object to every place where tracing is needed.


## Development

## Getting access to clojars.org

To be able to publish the library, please register on https://clojars.org and ask other team Automata members for access to this group:
https://clojars.org/groups/org.zalando.automata.

Follow this instruction to configure _~/.lein/credentials.clj.gpg_ on your machine: https://github.com/technomancy/leiningen/blob/master/doc/DEPLOY.md#gpg. Use `#"https://clojars.org/repo"` in your `credentials.clj` instead of `#"https://repo.clojars.org"`


### Trying the library locally without publishing

```
lein install
```

This will build the JAR and put it into your local `~/.m2`.
Then you can use it in your projects on the same computer.

### Letting your colleagues or CI try the library

Add `-SNAPSHOT` to the version and upload it to Clojars (without committing the changes):

```
lein deploy
```

You can use the published snapshot in a CI build of some other project to see if library changes don't break existing projects.

Please don't commit manually changed version, version should be managed automatically when doing `lein release` (read below)
and should look like "X.Y.Z" at all times.

### Releasing

Releasing follows the process described here: https://github.com/dryewo/clojure-library-template

In a nutshell:

1. Always describe changes being made in _CHANGELOG.md_.
2. Don't change version manually, version in _project.clj_ should be the latest released version
   - this also applies to the version in _README.md_.
   - CI will reject versions that don't look like "X.Y.Z".
3. Release only from master branch (after merging the PR).

To release the next version, check out master branch and run `lein release`,
depending on how big the changes are compared to the previously released version:

```
lein release :patch
# or
lein release :minor
# or
lein release :major
```

This will update _CHANGELOG.md_, _README.md_, _project.clj_, commit, tag, push and upload the JAR to Clojars.

### Troubleshooting
If you have a macbook, please install gnu-sed since puny default sed will not work.
In case of gpg being unable to sign, try

```bash
export GPG_TTY=$(tty)
```

## License

Copyright © 2019 Zalando SE

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.


[mount-lite]: https://github.com/aroemers/mount-lite
