(ns 
  #^{:doc "A simple event system that processes fired events in a thread pool."
     :author "Jeff Rose"}
  clj-event.core
  (:import (java.util.concurrent Executors LinkedBlockingQueue))
;  (:require [overtone.core.log :as log])
  (:use ;(overtone.core util)
        clojure.stacktrace
        [clojure.set :only [intersection difference]]))

(defn- arg-count
  "Get the arity of a function."
  [f]
  (let [m (first (.getDeclaredMethods (class f)))
        p (.getParameterTypes m)]
    (alength p)))

(defn- run-handler [handler & args]
  (try
    (apply handler (take (arg-count handler) args))
    (catch Exception e
      (comment
        (log/debug "Handler Exception - got args:" args"\n" (with-out-str
                                                                     (print-cause-trace e)))))))

(defn- cpu-count
  "Get the number of CPUs on this machine."
  []
  (.availableProcessors (Runtime/getRuntime)))


(def NUM-THREADS (cpu-count))
(defonce thread-pool (Executors/newFixedThreadPool NUM-THREADS))
(defonce event-handlers* (ref {}))

(defn on
  "Runs handler whenever events of type event-type are fired.  The handler can
  optionally except a single event argument, which is a map containing the
  :event-type property and any other properties specified when it was fired.

  (on ::booted #(do-stuff))
  (on ::midi-note-down (fn [event] (funky-bass (:note event))))

  Handlers can return :done to be removed from the handler list after execution."
  [event-type handler]
;  (log/debug "adding-handler for " event-type)
  (dosync
    (let [handlers (get @event-handlers* event-type #{})]
      (alter event-handlers* assoc event-type (conj handlers handler))
      true)))

(defn remove-handler
  "Remove an event handler previously registered to handle events of event-type.

  (defn my-foo-handler [event] (do-stuff (:val event)))

  (on ::foo my-foo-handler)
  (event ::foo :val 200) ; my-foo-handler gets called with {:event-type ::foo :val 200}
  (remove-handler ::foo my-foo-handler)
  (event ::foo :val 200) ; my-foo-handler no longer called
  "
  [event-type handler]
  (dosync
    (let [handlers (get @event-handlers* event-type #{})]
      (alter event-handlers* assoc event-type (difference handlers #{handler})))))

(defn clear-handlers
  "Remove all handlers for events of type event-type."
  [event-type]
  (dosync (alter event-handlers* dissoc event-type)))

(defn- handle-event
  "Runs the event handlers for the given event, and removes any handler that returns :done."
  [event]
  (let [event-type (:event-type event)
        handlers (get @event-handlers* event-type #{})
        keepers  (set (doall (filter #(not (= :done (run-handler % event))) handlers)))]
    (dosync (alter event-handlers* assoc event-type
                   (intersection keepers (get @event-handlers* event-type #{}))))))

(defn event
  "Fire an event of type event-type with any number of additional properties.
  NOTE: an event requires key/value pairs, and everything gets wrapped into an
  event map.  It will not work if you just pass values.

  (event ::my-event)
  (event ::filter-sweep-done :instrument :phat-bass)"
  [event-type & args]
  {:pre [(even? (count args))]}
;  (log/debug "firing event: " event-type args)
  (.execute thread-pool #(handle-event (apply hash-map :event-type event-type args))))
