(ns leiningen.release
  "Borrowed from https://github.com/relaynetwork/lein-release/blob/master/src/leiningen/release.clj"
  (:require [clojure.java.shell :as sh] [clojure.string :as string])
  (:import (java.util.regex Pattern)))

(defn raise [fmt & args]
  (throw (RuntimeException. (apply format fmt args))))

(def ^:dynamic config {})

(def commands
  {:add ["git" "add"]
   :commit ["git" "commit"]
   :pull-develop ["git" "pull" "origin" "develop"]
   :push ["git" "push"]
   :push-master ["git" "push" "origin" "master"]
   :push-develop ["git" "push" "origin" "develop"]
   :push-tags ["git" "push" "--tags"]
   :start-release ["git" "flow" "release" "start"]
   :status ["git" "status"]})

(defn git-flow-exists? []
  (-> (sh/sh "command" "-v" "git-flow")
      :exit
      zero?))

(defn sh! [& args]
  (let [res (apply sh/sh args)]
    (.println System/out (:out res))
    (.println System/err (:err res))
    (when-not (zero? (:exit res))
      (raise "Error: command failed %s => %s" args res))))

(defn scm! [cmd & args]
  (let [scm-cmd (get commands cmd)]
    (if-not scm-cmd
      (raise "No such SCM command: %s" cmd))
    (apply sh! (concat scm-cmd args))))

;; See: http://mojo.codehaus.org/versions-maven-plugin/version-rules.html
(defn compute-next-development-version [current-version]
  (let [parts             (vec (.split current-version "\\."))
        version-parts     (vec (take (dec (count parts)) parts))
        minor-version     (last parts)
        new-minor-version (str (inc (Integer/parseInt minor-version)) "-SNAPSHOT")]
    (string/join "." (conj version-parts new-minor-version))))

(defn replace-project-version [old-vstring new-vstring]
  (let [proj-file     (slurp "project.clj")
        new-proj-file (.replaceAll proj-file (format "\\(defproject .+? %s" old-vstring) new-vstring )
        matcher       (.matcher
                       (Pattern/compile (format "(\\(defproject .+? )\"\\Q%s\\E\"" old-vstring))
                       proj-file)]
    (if-not (.find matcher)
      (raise "Error: unable to find version string %s in project.clj file!" old-vstring))
    (.replaceFirst matcher (format "%s\"%s\"" (.group matcher 1) new-vstring))))

(defn set-project-version! [old-vstring new-vstring]
  (spit "project.clj" (replace-project-version old-vstring new-vstring)))

(defn extract-project-version-from-file
  ([] (extract-project-version-from-file "project.clj"))
  ([proj-file]
     (let [s (slurp proj-file)
           m (.matcher (Pattern/compile "\\(defproject .+? \"([^\"]+?)\"") s)]
       (if-not (.find m)
         (raise "Error: unable to find project version in file: %s" proj-file))
       (.group m 1))))

(defn is-snapshot? [vstring]
  (.endsWith vstring "-SNAPSHOT"))

(defn compute-release-version [current-version]
  (str (.replaceAll current-version "-SNAPSHOT" "")))

(defn finish-release! [version]
  (letfn [(sh-string! [s]
            (sh! "sh" "-c" (format "(%s)" s)))]
    (->> [(format "echo 'Releasing %s.' > .git/MY_TAGMSG" version)
          "git config core.editor \"mv .git/MY_TAGMSG\""
          (str "git flow release finish " version)
          "git config --unset core.editor"]
         (string/join "; ")
         (sh-string!))))

(defn release [project & args]
  (binding [config (or (:lein-release project) config)]
    (assert (git-flow-exists?) "git-flow is required for release. See https://github.com/nvie/gitflow for installation instructions.")
    (let [current-version  (get project :version)
          release-version  (compute-release-version current-version)
          next-dev-version (compute-next-development-version (.replaceAll current-version "-SNAPSHOT" ""))]
      (when (is-snapshot? current-version)
        (scm! :pull-develop)
        (scm! :start-release release-version)
        (println (format "setting project version %s => %s" current-version release-version))
        (set-project-version! current-version release-version)
        (println "adding, committing and tagging project.clj")
        (scm! :add "project.clj")
        (scm! :commit "-m" (format "lein-release plugin: preparing %s release" release-version))
        (finish-release! release-version))
      (when-not (is-snapshot? (extract-project-version-from-file))
        (println (format "updating version %s => %s for next dev cycle" release-version next-dev-version))
        (set-project-version! release-version next-dev-version)
        (scm! :add "project.clj")
        (scm! :commit "-m" (format "lein-release plugin: bumped version from %s to %s for next development cycle" release-version next-dev-version))
        (scm! :push-tags)
        (scm! :push-master)
        (scm! :push-develop)))))
