(ns leiningen.mig
  (:require [clojure.java.io :refer [as-file]]
            [clojure.string :as str]
            [mig
             [core :refer [extract-predicate find-migration-files load-files]]
             [pipe :refer [pipe]]]
            [slingshot.slingshot :refer [throw+ try+]]))

(def init (constantly {}))
(defn finish [ctx])

(def default-conf {:base-dir "migrations"
                   :directives [:up :down]
                   :init 'leiningen.mig/init
                   :finish 'leiningen.mig/finish})

(defn whats-applied
  "Lists files, like 'list-files' but with details of which of the
  directives' predicates pass."
  [ctx files args config]
  (let [fns (for [fl files
                  dir (map (comp symbol name) (:directives config))]
              (fn [ctx]
                (if-let [nctx ((extract-predicate fl dir) ctx)]
                  (do
                    (println (str fl " - " dir "-applied?: true"))
                    nctx)
                  (do
                    (println (str fl " - " dir "-applied?: false"))
                    ctx))))]
    (apply pipe ctx fns)))

(defn list-files
  "List of all files, applied or not, in the order they will be
  evaluated."
  [ctx files _ _]
  (doseq [fl files] (println (str fl))))

(defn apply-migs
  "Run through all migrations, applying those whose pred returns
  something falsey. Takes a direction, e.g. `up`."
  [ctx files args config]
  (let [dir (keyword (first args))
        dirs (:directives config)]
    (when (not (some #{dir} dirs))
      (throw+ {:type :mig
               :message (str "Expecting one of " (str/join ", " (map name dirs)))}))
    (load-files ctx files (symbol (name dir)))))

(def commands {:whats-applied {:args 0 :fn whats-applied}
               :list-files    {:args 0 :fn list-files}
               :apply         {:args 1 :fn apply-migs}})

(defn exec-command [args ctx files config]
  (let [cmd (keyword (first args))
        cmd-details (get commands cmd)]
    (when (not (= (count (rest args)) (:args cmd-details)))
      (throw+ {:type :mig
               :message (str cmd " expects " (:args cmd-details) " arguments.")}))
    ((:fn cmd-details) ctx files (rest args) config)))

(defn dyn-req [sym]
  (require (symbol (namespace sym)))
  (resolve sym))

(defn mig [project & args]
  (try+
   (let [config (merge default-conf (:mig project))
         ;; call the user-defined init function to get the initial
         ;; context for the migrations.
         ctx ((dyn-req (:init config)))
         files (find-migration-files
                (let [base (as-file (:base-dir config))]
                  (when-not (.exists base)
                    (throw+ {:type :mig
                             :message (str base " doesn't exist.")}))
                  base))
         finsih (dyn-req (:finish config))]
     (when (= (count args) 0)
       (throw+ {:type :mig
                :message "Expecting an argument."}))
     (finish (exec-command args ctx files config)))
   (catch [:type :mig] e
     (leiningen.core.main/exit 1 (:message e)))))
