(ns polvo.utils.deribit
  (:refer-clojure :exclude [send])
  (:require [clojure.data.json :as json]
            [clojure.string :as s]
            [clojure.math.combinatorics :refer [cartesian-product]]
            [clj-http.client :as http]
            [gniazdo.core :as ws]
            [buddy.core.mac :as mac]
            [buddy.core.codecs :as codecs]
            [ring.util.codec :refer [form-encode]]
            [camel-snake-kebab.core :as csk]
            [camel-snake-kebab.extras :as cske]
            [java-time :as time]))

(def ^{:private true :const true} api-ver "/api/v2/")

(def ^{:private true}
  endpoints {
             #{:websocket :production} "wss://www.deribit.com/ws"
             #{:websocket :test}       "wss://test.deribit.com/ws"
             #{:http :production}      "https://www.deribit.com"
             #{:http :test}            "https://test.deribit.com"})

(def ^{:const true}
  routes {;
          ; Authentication
          :auth                             "public/auth"
          :logout                           "private/logout"

          ; Session Management
          :set-heartbeat                    "public/heartbeat"
          :disable-heartbeat                "public/disable_heartbeat"
          :enable-cod                       "private/enable_cancel_on_disconnect"
          :disable-cod                      "private/disable_cancel_on_disconnect"

          ; Supporting
          :time                             "public/get_time"
          :hello                            "public/hello"
          :test                             "public/test"

          ; Subscription Management
          :public-subscribe                 "public/subscribe"
          :public-unsubscribe               "public/unsubscribe"
          :private-subscribe                "private/subscribe"
          :private-unsubscribe              "private/unsubscribe"

          ; Account Management
          :announcements                    "public/get_announcements"
          :subaccount-name                  "private/change_subaccount_name"
          :create-subaccount                "private/create_subaccount"
          :disable-subaccount-tfa           "private/disable_tfa_for_subaccount"
          :account-summary                  "private/get_account_summary"
          :get-email-language               "private/get_email_language"
          :new-announcements                "private/get_new_announcements"
          :position                         "private/get_position"
          :positions                        "private/get_positions"
          :subaccounts                      "private/get_subaccounts"
          :mark-as-read                     "private/set_announcement_as_read"
          :set-subaccount-email             "private/set_email_for_subaccount"
          :set-email-language               "private/set_email_language"
          :set-subaccount-password          "private/set_password_for_subaccount"
          :toggle-subaccount-notifications  "private/toggle_notifications_from_subaccount"
          :toggle-subaccount-login          "private/toggle_subaccount_login"

          ; Trading
          :buy                              "private/buy"
          :sell                             "private/sell"
          :edit                             "private/edit"
          :cancel                           "private/cancel"
          :cancel-all                       "private/cancel_all"
          :cancel-by-currency               "private/cancel_all_by_currency"
          :cancel-by-instrument             "private/cancel_all_by_instrument"
          :close-position                   "private/close_position"
          :margins                          "private/get_margins"
          :open-orders-by-currency          "private/get_open_orders_by_currency"
          :open-orders-by-instrument        "private/get_open_orders_by_instrument"
          :order-history-by-currency        "private/get_order_history_by_currency"
          :order-history-by-instrument      "private/get_order_history_by_instrument"
          :order-margins                    "private/get_order_margins_by_id"
          :order-state                      "private/get_order_state"
          :stop-order-history               "private/get_stop_order_history"
          :user-trades-by-currency          "private/get_user_trades_by_currency"
          :user-trades-by-currency-time     "private/get_user_trades_by_currency_and_time"
          :user-trades-by-instrument        "private/get_user_trades_by_instrument"
          :user-trades-by-instrument-time   "private/get_user_trades_by_instrument_and_time"
          :user-trades-by-order             "private/get_user_trades_by_order"
          :settlement-history-by-currency   "private/get_settlement_history_by_currency"
          :settlement-history-by-instrument "private/get_settlement_history_by_instrument"

          ; Market Data
          :book-summary-by-currency         "public/get_book_summary_by_currency"
          :book-summary-by-instrument       "public/get_book_summary_by_instrument"
          :contract-size                    "public/get_contract_size"
          :currencies                       "public/get_currencies"
          :funding-chart-data               "public/get_funding_chart_data"
          :historical-volatility            "public/get_historical_volatility"
          :index                            "public/get_index"
          :instruments                      "public/get_instruments"
          :last-settlements-by-currency     "public/get_settlements_by_currency"
          :last-settlements-by-instrument   "public/get_settlements_by_instrument"
          :last-trades-by-currency          "public/get_last_trades_by_currency"
          :last-trades-by-instrument        "public/get_last_trades_by_instrument"
          :last-trades-by-currency-time     "public/get_last_trades_by_currency_and_time"
          :last-trades-by-instrument-time   "public/get_last_trades_by_instrument_and_time"
          :order-book                       "public/get_order_book"
          :trade-volumes                    "public/get_trade_volumes"
          :tradingview-chart-data           "public/get_tradingview_chart_data"
          :ticker                           "public/ticker"

          ; Wallet Data
          :cancel-transfer                  "private/cancel_transfer_by_id"
          :cancel-withdrawal                "private/cancel_withdrawal"
          :create-deposit-address           "private/create_deposit_address"
          :current-deposit-address          "private/get_current_deposit_address"
          :deposits                         "private/get_deposits"
          :transfers                        "private/get_transfers"
          :withdrawals                      "private/get_withdrawals"
          :withdraw                         "private/withdraw"})

(defn- form-base [transport testing?]
  (endpoints #{transport (if testing? :test :production)}))

(defn- ->key-fn [as-is?] (if as-is? identity csk/->kebab-case-keyword))

(defn- gen-nonce []
  (let [letters (map char (range (int \a) (inc (int \z))))
        digits  (range 0 10)
        chars   (vec (concat letters digits))]
    (->> #(rand-nth chars) (repeatedly 8) (apply str))))

(defn- private? [url]
  (= (subs url 8 15) "private"))

(defn- http-auth [url key secret]
  (if (private? url)
    (let [body           ""
          request-data   (str "GET\n" url "\n" body "\n")
          timestamp      (System/currentTimeMillis)
          nonce          (gen-nonce)
          string-to-sign (str timestamp "\n" nonce "\n" request-data)
          signature      (-> string-to-sign
                             (mac/hash {:key secret
                                        :alg :hmac+sha256})
                             codecs/bytes->hex)]
      {"Authorization" (str "deri-hmac-sha256 id=" key
                            ",ts=" timestamp
                            ",sig=" signature
                            ",nonce=" nonce)})
    {}))

(defn http-request
  [route & {:keys [params testing? key secret as-is?]
            :or   {params {} testing? false key nil secret nil as-is? false}}]
  (let [base      (form-base :http testing?)
        query-str (->> params
                       (cske/transform-keys csk/->snake_case_string)
                       form-encode)
        url       (str api-ver (routes route) "?" query-str)
        req       {:headers (merge {"Content-Type" "application/json"}
                                   (http-auth url key secret))}
        key-fn    (->key-fn as-is?)]

    (-> (http/get (str base url) req)
        :body
        (json/read-str :key-fn key-fn)
        (get (key-fn "result")))))

(defn http-client
  "A closure over fn request allowing to provide default args for testing? api-key, api-secret and as-is?
  Returning function may be called with a param map or named args."
  [& options]
  (fn [route & [params-dict & _ :as params]]
    (let [p (if (map? params-dict)
              params-dict
              (apply hash-map params))]
      (apply http-request route :params p options))))

(defn ws-request
  [client route & {:keys [params id testing? key secret as-is?]
                   :or   {params {} id nil testing? false key nil secret nil as-is? false}}]
  (let [id-dict (if (nil? id) {} {:id id})
        msg     (-> {:jsonrpc "2.0"
                     :method  (routes route)
                     :params  (cske/transform-keys csk/->snake_case_string params)}
                    (merge id-dict))]
    (ws/send-msg client (json/write-str msg :escape-slash false))))

(defn ws-client
  [& {:keys [key secret handlers testing? as-is?]
      :or   {key nil secret nil handlers {} testing? false as-is? false}}]
  (let [url    (->> (str (form-base :websocket testing?) api-ver)
                    drop-last
                    (s/join ""))
        client (apply ws/connect url (apply concat handlers))]
    {:client client
     :opts   {:key      key
              :secret   secret
              :testing? testing?
              :as-is?   as-is?}}))

(defn send
  [deribit-ws route & [params & {:keys [id] :or {id nil}}]]
  (let [params (or params {})]
    (apply ws-request (:client deribit-ws) route (->> (:opts deribit-ws)
                                                      (merge {:id id})
                                                      (merge {:params params})
                                                      (apply concat)))))


(defn- is-interval? [x]
  (some #{:1m :3m :5m :15m :30m :1h :2h :4h :6h :8h :12h :1d :3d :1w :1M} [(keyword x)]))

(defn- as-symbol [x] (-> x name s/lower-case))

(def ^{:private true} kw->subscription
  {:price-index       "deribit_price_index"
   :price-ranking     "deribit_price_ranking"
   :markprice-options "markprice.options"
   :user-orders       "user.orders"
   :user-portfolio    "user.portfolio"
   :user-trades       "user.trades"})

(defn- matches->date [matches]
  (->> (str (format "%02d" (Integer. (matches 2))) " "
            (csk/->PascalCase (matches 3)) " " (matches 4))
       (time/local-date "dd MMM yy")
       (time/format "yyyy-MM-dd")))

(defn parse-instrument [instrument]
  (let [instrument      (s/upper-case instrument)
        future-regex    #"(BTC|ETH)-(\d{1,2})(JAN|FEB|MAR|APR|MAY|JUN|JUL|AUG|SEP|OCT|NOV|DEC)(\d{2})"
        perpetual-regex #"(BTC|ETH)-PERPETUAL"
        option-regex    #"(BTC|ETH)-(\d{1,2})(JAN|FEB|MAR|APR|MAY|JUN|JUL|AUG|SEP|OCT|NOV|DEC)(\d{2})-(\d+)-(C|P)"]
    (if-let [[_ currency & _ :as matches] (re-matches perpetual-regex instrument)]
      {:instrument instrument
       :kind       "perpetual"
       :currency   currency}
      (if-let [[_ currency & _ :as matches] (re-matches future-regex instrument)]
        {:instrument      instrument
         :kind            "future"
         :currency        currency
         :expiration-date (matches->date matches)}
        (if-let [[_ currency _ _ _ strike kind & _ :as matches] (re-matches option-regex instrument)]
          {:instrument      instrument
           :kind            "option"
           :currency        currency
           :expiration-date (matches->date matches)
           :strike          (Integer. strike)
           :option-type     ({"P" "put" "C" "call"} kind)}
          nil)))))

(defn- kw->snake_case [kw]
  (-> kw name s/lower-case csk/->snake_case_string))

(defn- vec->stream-name [channel & [index_name & _ :as args]]
  (let [subscription (if (contains? kw->subscription channel)
                       (kw->subscription channel)
                       (kw->snake_case channel))]
    (->> args
         (cons subscription)
         (map #(if (int? %) %
                            (if-let [instrument (parse-instrument (name %))]
                              (:instrument instrument)
                              (kw->snake_case %))))
         (s/join "."))))

(defn- expand-vec [t & args]
  (->> args
       (map #(if (coll? %) % [%]))
       (apply cartesian-product)
       (map #(apply vec->stream-name t %))))

(defn expand [x]
  (if (string? x) [x] (apply expand-vec x)))