/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EventListener;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.eclipse.jetty.util.IncludeExcludeSet;
import org.eclipse.jetty.util.MultiException;
import org.eclipse.jetty.util.component.AbstractLifeCycle;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class PathWatcher
extends AbstractLifeCycle
implements Runnable {
    private static final boolean IS_WINDOWS;
    static final Logger LOG;
    private static final WatchEvent.Kind<?>[] WATCH_EVENT_KINDS;
    private static final WatchEvent.Kind<?>[] WATCH_DIR_KINDS;
    private WatchService watchService;
    private WatchEvent.Modifier[] watchModifiers;
    private boolean nativeWatchService;
    private final List<Config> configs = new ArrayList<Config>();
    private final Map<WatchKey, Config> keys = new HashMap<WatchKey, Config>();
    private final List<EventListener> listeners = new CopyOnWriteArrayList<EventListener>();
    private final Map<Path, PathWatchEvent> pending = new LinkedHashMap<Path, PathWatchEvent>(32, 0.75f, false);
    private final List<PathWatchEvent> events = new ArrayList<PathWatchEvent>();
    private long updateQuietTimeDuration = 1000L;
    private TimeUnit updateQuietTimeUnit = TimeUnit.MILLISECONDS;
    private Thread thread;
    private boolean _notifyExistingOnStart = true;

    protected static <T> WatchEvent<T> cast(WatchEvent<?> event) {
        return event;
    }

    public Collection<Config> getConfigs() {
        return this.configs;
    }

    public void watch(Path file2) {
        Path abs = file2;
        if (!abs.isAbsolute()) {
            abs = file2.toAbsolutePath();
        }
        Config config = null;
        Path parent2 = abs.getParent();
        for (Config c : this.configs) {
            if (!c.getPath().equals(parent2)) continue;
            config = c;
            break;
        }
        if (config == null) {
            config = new Config(abs.getParent());
            config.addIncludeGlobRelative("");
            config.addIncludeGlobRelative(file2.getFileName().toString());
            this.watch(config);
        } else {
            config.addIncludeGlobRelative(file2.getFileName().toString());
        }
    }

    public void watch(Config config) {
        this.configs.add(config);
    }

    public void addListener(EventListener listener) {
        this.listeners.add(listener);
    }

    private void appendConfigId(StringBuilder s) {
        ArrayList<Path> dirs = new ArrayList<Path>();
        for (Config config : this.keys.values()) {
            dirs.add(config.path);
        }
        Collections.sort(dirs);
        s.append("[");
        if (dirs.size() > 0) {
            s.append(dirs.get(0));
            if (dirs.size() > 1) {
                s.append(" (+").append(dirs.size() - 1).append(")");
            }
        } else {
            s.append("<null>");
        }
        s.append("]");
    }

    @Override
    protected void doStart() throws Exception {
        this.createWatchService();
        this.setUpdateQuietTime(this.getUpdateQuietTimeMillis(), TimeUnit.MILLISECONDS);
        for (Config c : this.configs) {
            this.registerTree(c.getPath(), c, this.isNotifyExistingOnStart());
        }
        StringBuilder threadId = new StringBuilder();
        threadId.append("PathWatcher@");
        threadId.append(Integer.toHexString(this.hashCode()));
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} -> {}", this, threadId);
        }
        this.thread = new Thread((Runnable)this, threadId.toString());
        this.thread.setDaemon(true);
        this.thread.start();
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        if (this.watchService != null) {
            this.watchService.close();
        }
        this.watchService = null;
        this.thread = null;
        this.keys.clear();
        this.pending.clear();
        this.events.clear();
        super.doStop();
    }

    public void reset() {
        if (!this.isStopped()) {
            throw new IllegalStateException("PathWatcher must be stopped before reset.");
        }
        this.configs.clear();
        this.listeners.clear();
    }

    private void createWatchService() throws IOException {
        this.watchService = FileSystems.getDefault().newWatchService();
        WatchEvent.Modifier[] modifiers = null;
        boolean nativeService = true;
        try {
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            Class<?> pollingWatchServiceClass = Class.forName("sun.nio.fs.PollingWatchService", false, cl);
            if (pollingWatchServiceClass.isAssignableFrom(this.watchService.getClass())) {
                nativeService = false;
                LOG.info("Using Non-Native Java {}", pollingWatchServiceClass.getName());
                Class<?> c = Class.forName("com.sun.nio.file.SensitivityWatchEventModifier");
                Field f = c.getField("HIGH");
                modifiers = new WatchEvent.Modifier[]{(WatchEvent.Modifier)f.get(c)};
            }
        }
        catch (Throwable t) {
            LOG.ignore(t);
        }
        this.watchModifiers = modifiers;
        this.nativeWatchService = nativeService;
    }

    protected boolean isNotifiable() {
        return this.isStarted() || !this.isStarted() && this.isNotifyExistingOnStart();
    }

    public Iterator<EventListener> getListeners() {
        return this.listeners.iterator();
    }

    public long getUpdateQuietTimeMillis() {
        return TimeUnit.MILLISECONDS.convert(this.updateQuietTimeDuration, this.updateQuietTimeUnit);
    }

    private void registerTree(Path dir2, Config config, boolean notify) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("registerTree {} {} {}", dir2, config, notify);
        }
        if (!Files.isDirectory(dir2, new LinkOption[0])) {
            throw new IllegalArgumentException(dir2.toString());
        }
        this.register(dir2, config);
        MultiException me = new MultiException();
        try (Stream<Path> stream = Files.list(dir2);){
            stream.forEach(p -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("registerTree? {}", p);
                }
                try {
                    if (notify && config.test((Path)p)) {
                        this.pending.put((Path)p, new PathWatchEvent((Path)p, PathWatchEventType.ADDED, config));
                    }
                    switch (config.handleDir((Path)p)) {
                        case ENTER: {
                            this.registerTree((Path)p, config.asSubConfig((Path)p), notify);
                            break;
                        }
                        case WATCH: {
                            this.registerDir((Path)p, config);
                            break;
                        }
                    }
                }
                catch (IOException e2) {
                    me.add(e2);
                }
            });
        }
        try {
            me.ifExceptionThrow();
        }
        catch (IOException e2) {
            throw e2;
        }
        catch (Throwable th) {
            throw new IOException(th);
        }
    }

    private void registerDir(Path path2, Config config) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("registerDir {} {}", path2, config);
        }
        if (!Files.isDirectory(path2, new LinkOption[0])) {
            throw new IllegalArgumentException(path2.toString());
        }
        this.register(path2, config.asSubConfig(path2), WATCH_DIR_KINDS);
    }

    protected void register(Path path2, Config config) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Registering watch on {} {}", path2, this.watchModifiers == null ? null : Arrays.asList(this.watchModifiers));
        }
        this.register(path2, config, WATCH_EVENT_KINDS);
    }

    private void register(Path path2, Config config, WatchEvent.Kind<?>[] kinds) throws IOException {
        if (this.watchModifiers != null) {
            WatchKey key2 = path2.register(this.watchService, kinds, this.watchModifiers);
            this.keys.put(key2, config);
        } else {
            WatchKey key3 = path2.register(this.watchService, kinds);
            this.keys.put(key3, config);
        }
    }

    public boolean removeListener(Listener listener) {
        return this.listeners.remove(listener);
    }

    @Override
    public void run() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Starting java.nio file watching with {}", this.watchService);
        }
        long wait_time = this.getUpdateQuietTimeMillis();
        WatchService watch = this.watchService;
        while (this.isRunning() && this.thread == Thread.currentThread()) {
            try {
                WatchKey key2;
                long now = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
                for (Map.Entry<WatchKey, Config> e2 : this.keys.entrySet()) {
                    WatchKey k = e2.getKey();
                    Config c = e2.getValue();
                    if (c.isPaused(now) || k.reset()) continue;
                    this.keys.remove(k);
                    if (!this.keys.isEmpty()) continue;
                    return;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Waiting for poll({})", wait_time);
                }
                WatchKey watchKey = wait_time < 0L ? watch.take() : (key2 = wait_time > 0L ? watch.poll(wait_time, this.updateQuietTimeUnit) : watch.poll());
                while (key2 != null) {
                    this.handleKey(key2);
                    key2 = watch.poll();
                }
                wait_time = this.processPending();
                this.notifyEvents();
            }
            catch (ClosedWatchServiceException e3) {
                return;
            }
            catch (InterruptedException e4) {
                if (this.isRunning()) {
                    LOG.warn(e4);
                    continue;
                }
                LOG.ignore(e4);
            }
        }
    }

    private void handleKey(WatchKey key2) {
        Config config = this.keys.get(key2);
        if (config == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("WatchKey not recognized: {}", key2);
            }
            return;
        }
        for (WatchEvent<?> event : key2.pollEvents()) {
            WatchEvent<Path> ev = PathWatcher.cast(event);
            Path name2 = (Path)ev.context();
            Path path2 = config.resolve(name2);
            if (LOG.isDebugEnabled()) {
                LOG.debug("handleKey? {} {} {}", ev.kind(), config.toShortPath(path2), config);
            }
            if (ev.kind() == StandardWatchEventKinds.ENTRY_MODIFY && Files.exists(path2, new LinkOption[0]) && Files.isDirectory(path2, new LinkOption[0])) continue;
            if (config.test(path2)) {
                this.handleWatchEvent(path2, new PathWatchEvent(path2, ev, config));
            } else if (config.getRecurseDepth() == -1) {
                Path parent2 = path2.getParent();
                Config parentConfig = config.getParent();
                this.handleWatchEvent(parent2, new PathWatchEvent(parent2, PathWatchEventType.MODIFIED, parentConfig));
                continue;
            }
            if (ev.kind() != StandardWatchEventKinds.ENTRY_CREATE) continue;
            try {
                switch (config.handleDir(path2)) {
                    case ENTER: {
                        this.registerTree(path2, config.asSubConfig(path2), true);
                        break;
                    }
                    case WATCH: {
                        this.registerDir(path2, config);
                        break;
                    }
                }
            }
            catch (IOException e2) {
                LOG.warn(e2);
            }
        }
    }

    public void handleWatchEvent(Path path2, PathWatchEvent event) {
        PathWatchEvent existing = this.pending.get(path2);
        if (LOG.isDebugEnabled()) {
            LOG.debug("handleWatchEvent {} {} <= {}", path2, event, existing);
        }
        switch (event.getType()) {
            case ADDED: {
                if (existing != null && existing.getType() == PathWatchEventType.MODIFIED) {
                    this.events.add(new PathWatchEvent(path2, PathWatchEventType.DELETED, existing.getConfig()));
                }
                this.pending.put(path2, event);
                break;
            }
            case MODIFIED: {
                if (existing == null) {
                    this.pending.put(path2, event);
                    break;
                }
                existing.modified();
                break;
            }
            case DELETED: 
            case UNKNOWN: {
                if (existing != null) {
                    this.pending.remove(path2);
                }
                this.events.add(event);
            }
        }
    }

    private long processPending() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("processPending> {}", this.pending.values());
        }
        long now = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        long wait = Long.MAX_VALUE;
        for (PathWatchEvent event : new ArrayList<PathWatchEvent>(this.pending.values())) {
            Path path2 = event.getPath();
            if (this.pending.containsKey(path2.getParent())) continue;
            if (event.isQuiet(now, this.getUpdateQuietTimeMillis())) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("isQuiet {}", event);
                }
                this.pending.remove(path2);
                this.events.add(event);
                continue;
            }
            long ms_to_check = event.toQuietCheck(now, this.getUpdateQuietTimeMillis());
            if (LOG.isDebugEnabled()) {
                LOG.debug("pending {} {}", event, ms_to_check);
            }
            if (ms_to_check >= wait) continue;
            wait = ms_to_check;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("processPending< {}", this.pending.values());
        }
        return wait == Long.MAX_VALUE ? -1L : wait;
    }

    private void notifyEvents() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("notifyEvents {}", this.events.size());
        }
        if (this.events.isEmpty()) {
            return;
        }
        boolean event_listeners = false;
        for (EventListener listener : this.listeners) {
            if (listener instanceof EventListListener) {
                try {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("notifyEvents {} {}", listener, this.events);
                    }
                    ((EventListListener)listener).onPathWatchEvents(this.events);
                }
                catch (Throwable t) {
                    LOG.warn(t);
                }
                continue;
            }
            event_listeners = true;
        }
        if (event_listeners) {
            for (PathWatchEvent event : this.events) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("notifyEvent {} {}", event, this.listeners);
                }
                for (EventListener listener : this.listeners) {
                    if (!(listener instanceof Listener)) continue;
                    try {
                        ((Listener)listener).onPathWatchEvent(event);
                    }
                    catch (Throwable t) {
                        LOG.warn(t);
                    }
                }
            }
        }
        this.events.clear();
    }

    public void setNotifyExistingOnStart(boolean notify) {
        this._notifyExistingOnStart = notify;
    }

    public boolean isNotifyExistingOnStart() {
        return this._notifyExistingOnStart;
    }

    public void setUpdateQuietTime(long duration, TimeUnit unit) {
        long desiredMillis = unit.toMillis(duration);
        if (this.watchService != null && !this.nativeWatchService && desiredMillis < 5000L) {
            LOG.warn("Quiet Time is too low for non-native WatchService [{}]: {} < 5000 ms (defaulting to 5000 ms)", this.watchService.getClass().getName(), desiredMillis);
            this.updateQuietTimeDuration = 5000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        if (IS_WINDOWS && desiredMillis < 1000L) {
            LOG.warn("Quiet Time is too low for Microsoft Windows: {} < 1000 ms (defaulting to 1000 ms)", desiredMillis);
            this.updateQuietTimeDuration = 1000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        this.updateQuietTimeDuration = duration;
        this.updateQuietTimeUnit = unit;
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder(this.getClass().getName());
        this.appendConfigId(s);
        return s.toString();
    }

    static {
        String os = System.getProperty("os.name");
        if (os == null) {
            IS_WINDOWS = false;
        } else {
            String osl = os.toLowerCase(Locale.ENGLISH);
            IS_WINDOWS = osl.contains("windows");
        }
        LOG = Log.getLogger(PathWatcher.class);
        WATCH_EVENT_KINDS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
        WATCH_DIR_KINDS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE};
    }

    public static class PathMatcherSet
    extends HashSet<PathMatcher>
    implements Predicate<Path> {
        @Override
        public boolean test(Path path2) {
            for (PathMatcher pm : this) {
                if (!pm.matches(path2)) continue;
                return true;
            }
            return false;
        }
    }

    private static class ExactPathMatcher
    implements PathMatcher {
        private final Path path;

        ExactPathMatcher(Path path2) {
            this.path = path2;
        }

        @Override
        public boolean matches(Path path2) {
            return this.path.equals(path2);
        }
    }

    public static enum PathWatchEventType {
        ADDED,
        DELETED,
        MODIFIED,
        UNKNOWN;

    }

    public class PathWatchEvent {
        private final Path path;
        private final PathWatchEventType type;
        private final Config config;
        long checked;
        long modified;
        long length;

        public PathWatchEvent(Path path2, PathWatchEventType type2, Config config) {
            this.path = path2;
            this.type = type2;
            this.config = config;
            this.checked = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
            this.check();
        }

        public Config getConfig() {
            return this.config;
        }

        public PathWatchEvent(Path path2, WatchEvent<Path> event, Config config) {
            this.path = path2;
            this.type = event.kind() == StandardWatchEventKinds.ENTRY_CREATE ? PathWatchEventType.ADDED : (event.kind() == StandardWatchEventKinds.ENTRY_DELETE ? PathWatchEventType.DELETED : (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY ? PathWatchEventType.MODIFIED : PathWatchEventType.UNKNOWN));
            this.config = config;
            this.checked = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
            this.check();
        }

        private void check() {
            if (Files.exists(this.path, new LinkOption[0])) {
                try {
                    this.modified = Files.getLastModifiedTime(this.path, new LinkOption[0]).toMillis();
                    this.length = Files.size(this.path);
                }
                catch (IOException e2) {
                    this.modified = -1L;
                    this.length = -1L;
                }
            } else {
                this.modified = -1L;
                this.length = -1L;
            }
        }

        public boolean isQuiet(long now, long quietTime) {
            long lastModified = this.modified;
            long lastLength = this.length;
            this.check();
            if (lastModified == this.modified && lastLength == this.length) {
                return now - this.checked >= quietTime;
            }
            this.checked = now;
            return false;
        }

        public long toQuietCheck(long now, long quietTime) {
            long check2 = quietTime - (now - this.checked);
            if (check2 <= 0L) {
                return quietTime;
            }
            return check2;
        }

        public void modified() {
            long now;
            this.checked = now = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
            this.check();
            this.config.setPauseUntil(now + PathWatcher.this.getUpdateQuietTimeMillis());
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            PathWatchEvent other = (PathWatchEvent)obj;
            if (this.path == null ? other.path != null : !this.path.equals(other.path)) {
                return false;
            }
            return this.type == other.type;
        }

        public Path getPath() {
            return this.path;
        }

        public PathWatchEventType getType() {
            return this.type;
        }

        @Deprecated
        public int getCount() {
            return 1;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.path == null ? 0 : this.path.hashCode());
            result = 31 * result + (this.type == null ? 0 : this.type.hashCode());
            return result;
        }

        public String toString() {
            return String.format("PathWatchEvent[%8s|%s]", new Object[]{this.type, this.path});
        }
    }

    public static interface EventListListener
    extends EventListener {
        public void onPathWatchEvents(List<PathWatchEvent> var1);
    }

    public static interface Listener
    extends EventListener {
        public void onPathWatchEvent(PathWatchEvent var1);
    }

    public static enum DirAction {
        IGNORE,
        WATCH,
        ENTER;

    }

    public static class Config
    implements Predicate<Path> {
        public static final int UNLIMITED_DEPTH = -9999;
        private static final String PATTERN_SEP;
        protected final Config parent;
        protected final Path path;
        protected final IncludeExcludeSet<PathMatcher, Path> includeExclude;
        protected int recurseDepth = 0;
        protected boolean excludeHidden = false;
        protected long pauseUntil;

        public Config(Path path2) {
            this(path2, null);
        }

        public Config(Path path2, Config parent2) {
            IncludeExcludeSet<PathMatcher, Path> includeExcludeSet;
            this.parent = parent2;
            if (parent2 == null) {
                IncludeExcludeSet<PathMatcher, Path> includeExcludeSet2;
                includeExcludeSet = includeExcludeSet2;
                super(PathMatcherSet.class);
            } else {
                includeExcludeSet = parent2.includeExclude;
            }
            this.includeExclude = includeExcludeSet;
            Path dir2 = path2;
            if (!Files.exists(path2, new LinkOption[0])) {
                throw new IllegalStateException("Path does not exist: " + path2);
            }
            if (!Files.isDirectory(path2, new LinkOption[0])) {
                dir2 = path2.getParent();
                this.includeExclude.include(new ExactPathMatcher(path2));
                this.setRecurseDepth(0);
            }
            this.path = dir2;
        }

        public Config getParent() {
            return this.parent;
        }

        public void setPauseUntil(long time2) {
            if (time2 > this.pauseUntil) {
                this.pauseUntil = time2;
            }
        }

        public boolean isPaused(long now) {
            if (this.pauseUntil == 0L) {
                return false;
            }
            if (this.pauseUntil > now) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("PAUSED {}", this);
                }
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("unpaused {}", this);
            }
            this.pauseUntil = 0L;
            return false;
        }

        public void addExclude(PathMatcher matcher) {
            this.includeExclude.exclude(matcher);
        }

        public void addExclude(String syntaxAndPattern) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding exclude: [{}]", syntaxAndPattern);
            }
            this.addExclude(this.path.getFileSystem().getPathMatcher(syntaxAndPattern));
        }

        public void addExcludeGlobRelative(String pattern) {
            this.addExclude(this.toGlobPattern(this.path, pattern));
        }

        public void addExcludeHidden() {
            if (!this.excludeHidden) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Adding hidden files and directories to exclusions", new Object[0]);
                }
                this.excludeHidden = true;
            }
        }

        public void addExcludes(List<String> syntaxAndPatterns) {
            for (String syntaxAndPattern : syntaxAndPatterns) {
                this.addExclude(syntaxAndPattern);
            }
        }

        public void addInclude(PathMatcher matcher) {
            this.includeExclude.include(matcher);
        }

        public void addInclude(String syntaxAndPattern) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding include: [{}]", syntaxAndPattern);
            }
            this.addInclude(this.path.getFileSystem().getPathMatcher(syntaxAndPattern));
        }

        public void addIncludeGlobRelative(String pattern) {
            this.addInclude(this.toGlobPattern(this.path, pattern));
        }

        public void addIncludes(List<String> syntaxAndPatterns) {
            for (String syntaxAndPattern : syntaxAndPatterns) {
                this.addInclude(syntaxAndPattern);
            }
        }

        public Config asSubConfig(Path dir2) {
            Config subconfig = new Config(dir2, this);
            if (dir2 == this.path) {
                throw new IllegalStateException("sub " + dir2.toString() + " of " + this);
            }
            subconfig.recurseDepth = this.recurseDepth == -9999 ? -9999 : this.recurseDepth - (dir2.getNameCount() - this.path.getNameCount());
            if (LOG.isDebugEnabled()) {
                LOG.debug("subconfig {} of {}", subconfig, this.path);
            }
            return subconfig;
        }

        public int getRecurseDepth() {
            return this.recurseDepth;
        }

        public boolean isRecurseDepthUnlimited() {
            return this.recurseDepth == -9999;
        }

        public Path getPath() {
            return this.path;
        }

        public Path resolve(Path path2) {
            if (Files.isDirectory(this.path, new LinkOption[0])) {
                return this.path.resolve(path2);
            }
            if (Files.exists(this.path, new LinkOption[0])) {
                return this.path;
            }
            return path2;
        }

        @Override
        public boolean test(Path path2) {
            int depth;
            if (this.excludeHidden && this.isHidden(path2)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("test({}) -> [Hidden]", this.toShortPath(path2));
                }
                return false;
            }
            if (!path2.startsWith(this.path)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("test({}) -> [!child {}]", this.toShortPath(path2), this.path);
                }
                return false;
            }
            if (this.recurseDepth != -9999 && (depth = path2.getNameCount() - this.path.getNameCount() - 1) > this.recurseDepth) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("test({}) -> [depth {}>{}]", this.toShortPath(path2), depth, this.recurseDepth);
                }
                return false;
            }
            boolean matched = this.includeExclude.test(path2);
            if (LOG.isDebugEnabled()) {
                LOG.debug("test({}) -> {}", this.toShortPath(path2), matched);
            }
            return matched;
        }

        public void setRecurseDepth(int depth) {
            this.recurseDepth = depth;
        }

        private String toGlobPattern(Path path2, String subPattern) {
            StringBuilder s = new StringBuilder();
            s.append("glob:");
            boolean needDelim = false;
            Path root2 = path2.getRoot();
            if (root2 != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Path: {} -> Root: {}", path2, root2);
                }
                for (Object c : (Object)root2.toString().toCharArray()) {
                    if (c == 92) {
                        s.append(PATTERN_SEP);
                        continue;
                    }
                    s.append((char)c);
                }
            } else {
                needDelim = true;
            }
            for (Path segment : path2) {
                if (needDelim) {
                    s.append(PATTERN_SEP);
                }
                s.append(segment);
                needDelim = true;
            }
            if (subPattern != null && subPattern.length() > 0) {
                if (needDelim) {
                    s.append(PATTERN_SEP);
                }
                for (Object c : (Object)subPattern.toCharArray()) {
                    if (c == 47) {
                        s.append(PATTERN_SEP);
                        continue;
                    }
                    s.append((char)c);
                }
            }
            return s.toString();
        }

        DirAction handleDir(Path path2) {
            try {
                if (!Files.isDirectory(path2, new LinkOption[0])) {
                    return DirAction.IGNORE;
                }
                if (this.excludeHidden && this.isHidden(path2)) {
                    return DirAction.IGNORE;
                }
                if (this.getRecurseDepth() == 0) {
                    return DirAction.WATCH;
                }
                return DirAction.ENTER;
            }
            catch (Exception e2) {
                LOG.ignore(e2);
                return DirAction.IGNORE;
            }
        }

        public boolean isHidden(Path path2) {
            try {
                if (!path2.startsWith(this.path)) {
                    return true;
                }
                for (int i = this.path.getNameCount(); i < path2.getNameCount(); ++i) {
                    if (!path2.getName(i).toString().startsWith(".")) continue;
                    return true;
                }
                return Files.exists(path2, new LinkOption[0]) && Files.isHidden(path2);
            }
            catch (IOException e2) {
                LOG.ignore(e2);
                return false;
            }
        }

        public String toShortPath(Path path2) {
            if (!path2.startsWith(this.path)) {
                return path2.toString();
            }
            return this.path.relativize(path2).toString();
        }

        public String toString() {
            StringBuilder s = new StringBuilder();
            s.append(this.path).append(" [depth=");
            if (this.recurseDepth == -9999) {
                s.append("UNLIMITED");
            } else {
                s.append(this.recurseDepth);
            }
            s.append(']');
            return s.toString();
        }

        static {
            String sep = File.separator;
            if (File.separatorChar == '\\') {
                sep = "\\\\";
            }
            PATTERN_SEP = sep;
        }
    }
}

