(ns leiningen.polylith.cmd
  (:require [clojure.pprint :as p]
            [clojure.string :as str]
            [leiningen.polylith.core :as core]
            [leiningen.polylith.file :as file]
            [leiningen.polylith.info :as info]))

(defn help []
  (println "Polylith 0.0.12-alpha (2017-11-30)")
  (println "  https://github.com/tengstrand/polylith")
  (println)
  (println "  lein polylith x     (where x is):")
  (println)
  (println "    changes x s1 s2   List changed components, systems or builds")
  (println "                      between two Git sha1:s")
  (println "    deps              List all dependencies")
  (println "    help              Show this help")
  (println "    info x [s1 s2]    list systems, components and builds")
  (println "    settings          The polylith settings in current project.clj")
  (println "    tests x [s1 s2]   Show or run tests"))

(defn info [root-dir [cmd last-success-sha1 current-sha1]]
  (if (nil? cmd)
    (do
      (println "Missing parameters, use the format:")
      (println "   lein polylith info x [s1 s2]")
      (println "     x = a -> show all")
      (println "         c -> show changes")
      (println "         + -> show Git diff")
      (println "         a+ -> show all + diff")
      (println "         c+ -> show changes + diff")
      (println "     s1 = last successful Git sha1")
      (println "     s2 = current Git sha1")
      (println)
      (println "   examples:")
      (println "     lein polylith info a+")
      (println "     lein polylith info c 1c5196cb4a0aa5f30c8ac52220614e959440e37b 8dfb454c5ed7849b52991335be1a794d591671dd"))
    (let [[show-changed?
           show-unchanged?
           show-diff?] (condp = cmd
                         "a" [true true false]
                         "a+" [true true true]
                         "+" [false false true]
                         "c" [true false false]
                         "c+" [true false true]
                         [false false false])
           data (if (and last-success-sha1 current-sha1)
                  (core/info root-dir last-success-sha1 current-sha1)
                  (core/info root-dir))]
      (when (or show-changed? show-unchanged?)
        (info/print-info data show-unchanged?))
      (when show-diff?
        (println "changes:")
        (doseq [path (:diff data)]
          (println " " path))))))

(defn deps [root-dir]
  (doseq [dependency (core/all-dependencies root-dir)]
    (println dependency)))

(defn changes [root-dir [cmd last-success-sha1 current-sha1]]
  (if (or (nil? current-sha1)
          (nil? last-success-sha1))
    (do
      (println "Missing parameters, use the format:")
      (println "   lein polylith changes x s1 s2")
      (println "     x = b -> show changed builds")
      (println "         s -> show changed systems")
      (println "         c -> show changed components")
      (println "     s1 = last successful Git sha1")
      (println "     s2 = current Git sha1")
      (println)
      (println "   example:")
      (println "     lein polylith changes s 1c5196cb4a0aa5f30c8ac52220614e959440e37b 8dfb454c5ed7849b52991335be1a794d591671dd"))
    (doseq [dir (core/changes root-dir cmd last-success-sha1 current-sha1)]
      (println (str " " dir)))))

(defn task-not-found [subtask]
  (println "Subtask" subtask "not found.")
  (help))

(defn settings [root-dir settings]
  (println "root-dir:")
  (println " " root-dir)
  (println "settings:")
  (println " " settings))

(defn tests [root-dir [cmd last-success-sha1 current-sha1]]
  (if (nil? cmd)
    (do
      (println "Missing parameters, use the format:")
      (println "   lein polylith tests x [s1 s2]")
      (println "     x = a -> show tests to run")
      (println "         u -> show unit tests to run")
      (println "         i -> show integration tests to run")
      (println "         a+ -> execute all tests")
      (println "         u+ -> execute unit tests")
      (println "         i+ -> execute integration tests")
      (println "     s1 = last successful Git sha1")
      (println "     s2 = current Git sha1")
      (println)
      (println "   examples:")
      (println "     lein polylith tests u")
      (println "     lein polylith tests xa 1c5196cb4a0aa5f30c8ac52220614e959440e37b 8dfb454c5ed7849b52991335be1a794d591671dd"))
    (let [[show? run-type] (condp = cmd
                             "a" [true [true true]]
                             "u" [true [true false]]
                             "i" [true [false true]]
                             "a+" [false [true true]]
                             "u+" [false [true false]]
                             "i+" [false [false true]]
                             [false [false false]])
          tests (if (and last-success-sha1 current-sha1)
                  (core/tests root-dir run-type [last-success-sha1 current-sha1])
                  (core/tests root-dir run-type))]
      (if show?
        (core/show-tests tests)
        (core/run-tests tests)))))

;(tests "/Users/joakimtengstrand/IdeaProjects/project-unicorn"
;       ["u+"
;        "59977793c809b3e9a9ae6fee1c8029643b7034b5"
;        "6f54526fca154d6d2e0b55f80b91269995177cec"])
