(ns pulley.ring
  (:require [pulley.store                   :as store]
            [ring.util.response             :as response]
            [ring.middleware.params         :as ring-params]
            [ring.middleware.keyword-params :as ring-keyword-params]
            [clojure.string                 :as string]
            [clj-time.format                :as time-format]
            [cemerick.url                   :as url])
  (:import [java.net URL]))



(defn- parse-long
  [v]
  (when v
    (try
      (Long/parseLong v)
      (catch NumberFormatException e))))

(defn- scheme
  "returns the scheme as a string"
  [{:keys [scheme headers]}]
  (or (and (= (get headers "https") "on") "https")
      (and (= (get headers "x-forwarded-ssl") "on") "https")
      (get headers "x-forwarded-scheme")
      (when-let [proto (get headers "x-forwarded-proto")]
        (first (string/split proto #",")))
      (name scheme)))

(defn- host-with-port
  [{:keys [server-name] :as req}]
  (or
      (get-in req [:headers "host"])
      server-name))

(defn- feed-url
  [base-url {:keys [uri] :as req} from]
  (let [scheme-with-host (or base-url
                             (str (scheme req) "://" (host-with-port req)))]
    (str scheme-with-host
         uri
         "?from=" from)))

(defn- full-page?
  [page-size events]
  (= page-size (count events)))

(defn- etag
  [from events]
  (if (< 0 (count events))
    (str from "-" (+ from (count events)))
    (str from)))

(defn- response-map
  [req from page-size base-url events]
  (cond-> {}

          (not (empty? events))
          (assoc :events (vec events))

          (< 0 (count events))
          (assoc :next (feed-url base-url req (+ from (count events))))))


(defn- response-headers
  [req from page-size events]
  (cond-> {"Content-Type" "application/edn;charset=UTF8"}

          (< 0 (count events))
          (assoc "ETag" (etag from events))

          (full-page? page-size events)
          (assoc "Cache-Control" "max-age=31536000")))

(defn- events-response
  [store page-size base-url req]
  (let [from   (-> req :params :from parse-long (or 1) (max 1))
        events (store/events-from store
                                  from
                                  page-size)]
    {:status  200
     :headers (response-headers req from page-size events)
     :body    (pr-str (response-map req from page-size base-url events))}))

(defn- timestamp-response
  [store base-url req]
  (let [unix-time (-> req :params :from-timestamp parse-long (or 0) (max 1))
        id        (store/resolve-timestamp store unix-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url base-url req id)}}
      {:status 404})))

(defn- date-time-response
  [store base-url req]
  (let [date-time (->> req
                       :params
                       :from-date-time
                       time-format/parse)
        id        (store/resolve-date-time store date-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url base-url req id)}}
      {:status 404})))

(defn- make-handler
  [store page-size base-url]
  (fn [req]
    (cond
     (-> req :params :from)           (events-response store page-size base-url req)
     (-> req :params :from-timestamp) (timestamp-response store base-url req)
     (-> req :params :from-date-time) (date-time-response store base-url req)
     :default                         (events-response store page-size base-url req))))

(defn handler
  [store & {:keys [page-size base-url] :or {page-size 100}}]
  (-> (make-handler store page-size)

      (ring-keyword-params/wrap-keyword-params)
      (ring-params/wrap-params)))
