(ns pulley.store.mysql
  (:require [clojure.java.jdbc     :as jdbc]
            [pulley.store          :as store]))

;;--------------------------------------------------------------------
;; schema

(defn- table-exists?
  [table-name]
  (jdbc/with-query-results res
    ["SHOW TABLES LIKE ?" table-name]
    (pos? (count res))))

(defn- create-table
  [table-name]
  (jdbc/do-commands
    (str "CREATE TABLE `" table-name "` (
             `id` int(20) NOT NULL AUTO_INCREMENT,
             `timestamp` int(11) NOT NULL,
             `data` blob NOT NULL,
             PRIMARY KEY (`id`),
             KEY `timestamp` (`timestamp`)
           ) ENGINE=InnoDB DEFAULT CHARSET=latin1;")))

(defn- ensure-schema
  [db table-name]
  (jdbc/with-connection db
    (when-not (table-exists? table-name)
      (create-table table-name)))) 

;;--------------------------------------------------------------------
;; sql statements

(defn- insert-event
  [table-name event-data]
  (let [res (jdbc/insert-record table-name event-data)]
    (-> res :generated_key)))

(defn- events-from
  [table-name id count]
  (jdbc/with-query-results res
    [(str "SELECT id, timestamp, data
           FROM `" table-name "`
           WHERE id >= ? AND id < ?")
     id (+ id count)]
    (vec res)))

(defn- timestamp->id
  [table-name timestamp]
  (jdbc/with-query-results res
    [(str "SELECT MIN(id) AS `id` FROM `" table-name "`"
          "WHERE timestamp >= ?")
     timestamp]
    (-> res first :id)))

(defn event-store
  [table-name]
  (reify store/EventStore
    (-write
      [store event]
      (insert-event table-name event))
    (-events-from
      [store id count]
      (events-from table-name id count))
    (-timestamp->id
      [store timestamp]
      (timestamp->id table-name timestamp))))

(defn make-event-store
  ([db table-name]
     (make-event-store db table-name {}))
  ([db table-name opts]
     (when (:ensure-schema opts)
       (ensure-schema db table-name))
     (event-store table-name)))

(comment

  (def mysql-db {:subprotocol "mysql"
                 :subname "//127.0.0.1:3306/events_test"
                 :user "root"})

  
  
  (jdbc/with-connection mysql-db
    (table-exists? "events"))

  (jdbc/with-connection mysql-db
    (create-table "events42"))

  (jdbc/with-connection mysql-db
    (insert-event "events42" {:event-id 42}))

  (jdbc/with-connection mysql-db
    (events-from "events42" 1 10))

  (jdbc/with-connection mysql-db
    (timestamp->id "events42" 1368718882))

  )
