(ns reaction.jackdaw.processor
  "Lightweight component wrapper around jackdaw Streams (DSL) processor app."
  (:require
   [clojure.spec.alpha :as s]
   [com.stuartsierra.component :as component]
   [duct.logger :as log]
   [integrant.core :as ig]
   [jackdaw.streams :as streams]
   [jackdaw.streams.mock :as mock])
  (:import
   [org.apache.kafka.streams KafkaStreams Topology]))


(s/def ::topic-registry any?)
(s/def ::topology-builder-fn any?)
(s/def ::app-config (s/map-of string? string?))

(s/def ::processor
  (s/keys :req-un [::app-config ::topic-registry ::topology-builder-fn]))


(def app-config-defaults
  "Default Kafka Streams app configuration.
  Sets the `default.deserialization.exception.handler` to log and continue. This
  may be opinionated, but it seems like a saner default than the real default
  (which stops the app when a deserialization error occurs). One can always
  override these defaults via custom app-config passed to the component."
  {"default.deserialization.exception.handler"
   "org.apache.kafka.streams.errors.LogAndContinueExceptionHandler"})

(defn apply-comp
  "Returns fn or a comp of fns."
  [fn-or-fns]
  (if (sequential? fn-or-fns)
    (apply comp (vec fn-or-fns))
    fn-or-fns))

(defn kafka-streams*
  "A custom function to create a KafkaStreams application. This is similar to
  the one provided by the jackdaw library, except that it requires a built
  `Topology` instead of a Builder. Thus the caller is responsible for building
  the `Topology` and has access to it."
  [topology opts]
  (let [props (java.util.Properties.)]
    (.putAll props opts)
    (KafkaStreams. ^Topology topology
                   ^java.util.Properties props)))

(defn init-streams-app
  "Helper function to create a streams app with configuration."
  [topology app-config]
  (kafka-streams* topology (merge app-config-defaults app-config)))

(defrecord Processor [app-config topic-registry topology-builder-fn]
  component/Lifecycle
  (start [{:keys [app app-config topology] :as this}]
    ;; defensively create a new app if it doesn't exist.
    ;; this may be true on restart.
    (let [app (or app (init-streams-app topology app-config))]
      (assoc this :app (streams/start app))))
  (stop [{:keys [app] :as this}]
    (when app (streams/close app))
    (dissoc this :app)))

(defn processor
  "Creates a new Processor record."
  [app-config topic-registry topology-builder-fn]
  (let [topology-builder-fn* (apply-comp topology-builder-fn)
        builder (-> (streams/streams-builder)
                    topology-builder-fn*
                    streams/streams-builder*)
        topology (.build builder)
        app (init-streams-app topology app-config)]
    (assoc (->Processor app-config topic-registry topology-builder-fn)
           :app app
           :builder builder
           ;; Adding a complementary topic-configs because it was there in the
           ;; previously used library, but would like to remove this when
           ;; verified that it's not necessary.
           :topic-configs (:topic-configs topic-registry)
           :topology topology)))

(defn cleanup!
  "Delete the local state store directory given an unstarted Processor record.
  Will throw an exception when the Streams app is currently running."
  [{:keys [app app-config topology]}]
  (let [app (or app (init-streams-app topology app-config))]
    (.cleanUp ^KafkaStreams app)))

;;
;; A MockProcessor for unit testing a processor in isolation.
;;

(defrecord MockProcessor [topic-registry topology-builder-fn]
  component/Lifecycle
  (start [{:keys [topology-builder-fn] :as this}]
    (let [topology-builder-fn* (apply-comp topology-builder-fn)
          builder (-> (mock/streams-builder)
                      topology-builder-fn*)
          driver (mock/streams-builder->test-driver builder)]
      (assoc this :driver driver)))
  (stop [{:keys [driver] :as this}]
    (when driver (.stop driver))
    (dissoc this :driver)))

(defn mock-processor
  "Creates a new MockProcessor record."
  [topic-registry topology-builder-fn]
  (->MockProcessor topic-registry topology-builder-fn))

;;
;; Helpers for working with a mock processor
;;
(defn mock-produce!
  "Produce a KV to a specific topic of a mock processor."
  ([mock-processor topic-kw k v]
   (mock-produce! mock-processor topic-kw nil k v))
  ([mock-processor topic-kw time-ms k v]
   (let [args [(:driver mock-processor)
               (get-in mock-processor [:topic-registry :topic-configs topic-kw])]
         args (cond-> args time-ms (conj time-ms))
         args (conj args k v)]
     (apply mock/publish args))))

(defn mock-get-keyvals
  "Get KV pairs from a specific output topic of a mock processor."
  [mock-processor topic-kw]
  (mock/get-keyvals
    (:driver mock-processor)
    (get-in mock-processor [:topic-registry :topic-configs topic-kw])))


;;
;; Integrant multimethods.
;;

;: This function is available in Integrant core but marked private.
(defn normalize-key [k]
  (if (vector? k) (last k) k))

(defn log-lifecycle
  "Logs the start of a component. Logs the most specific key when a composite
  key is provided."
  [_ logger action system-key]
  (when logger (log/report logger action (normalize-key system-key))))

(defmethod ig/pre-init-spec :reaction.jackdaw/processor [_] ::processor)

(defmethod ig/init-key :reaction.jackdaw/processor
  [system-key {:keys [app-config logger topic-registry topology-builder-fn]}]
  (doto (processor app-config topic-registry topology-builder-fn)
    component/start
    (log-lifecycle logger :component-started system-key)))

(defmethod ig/halt-key! :reaction.jackdaw/processor
  [system-key {:keys [logger] :as this}]
  (when this
    (doto this
      (component/stop)
      (log-lifecycle logger :component-stopped system-key))))

(defmethod ig/init-key :reaction.jackdaw.processor/mock
  [_ {:keys [topic-registry topology-builder-fn]}]
  (-> (mock-processor topic-registry topology-builder-fn)
      component/start))

(defmethod ig/halt-key! :reaction.jackdaw.processor/mock
  [_ this]
  (when this (component/stop this)))
