(ns reaction.jackdaw.topic-registry
  (:require
    [clojure.spec.alpha :as s]
    [com.stuartsierra.component :as component]
    [integrant.core :as ig]
    [reaction.jackdaw.resolver :as resolver]))

(s/def ::topic-metadata
  (s/map-of keyword? (s/merge :jackdaw.serde-client/topic
                              :jackdaw.creation-client/topic)))

(s/def ::topic-registry (s/keys :req-un [::topic-metadata]))

;; `topic-metadata` is a map of "unresolved" topic metadata.
(defrecord BaseTopicRegistry [topic-metadata serde-resolver-fn]
  component/Lifecycle
  (start [this]
    (assoc this
           :topic-configs
           (resolver/resolve-topics topic-metadata serde-resolver-fn)))
  (stop [this]
    this))

;;
;; Note that `type-registry` may be `nil` or left unspecified.
;; In such a case jackdaw's default type registry will be used.
;;

(defn map->TopicRegistry
  [{:keys [topic-metadata schema-registry-url type-registry]}]
  (map->BaseTopicRegistry {:topic-metadata topic-metadata
                           :serde-resolver-fn (resolver/serde-resolver schema-registry-url type-registry)}))

(defn map->MockTopicRegistry
  [{:keys [topic-metadata type-registry]}]
  (map->BaseTopicRegistry {:topic-metadata topic-metadata
                           :serde-resolver-fn (resolver/mock-serde-resolver type-registry)}))


;;
;; Integrant methods.
;;

(defmethod ig/init-key :reaction.jackdaw/topic-registry
  [_ opts]
  (-> opts map->TopicRegistry component/start))

(defmethod ig/halt-key! :reaction.jackdaw/topic-registry
  [_ this]
  (component/stop this))

(defmethod ig/init-key :reaction.jackdaw/mock-topic-registry
  [_ opts]
  (-> opts map->MockTopicRegistry component/start))

(defmethod ig/halt-key! :reaction.jackdaw/mock-topic-registry
  [_ this]
  (component/stop this))
