(ns borg.client
  (:require [borg.transport.core :as transport])
  (:refer-clojure :exclude [get do set]))

(def ^:dynamic *client* nil)

(defn internal-do [client handler & [options]]
  (-> (transport/run-handler client handler options)
      (:details)))

(defn set-transporter! [trans-key]
  (transport/set-transporter! trans-key))

(defn connect [host port]
  (transport/client-create host port))

(defn disconnect
  ([]
   (disconnect *client*))
  ([client]
    (transport/client-close client)))

(defmacro with-client [client & body]
  `(binding [*client* ~client]
     ~@body))

(defn do
  "Run a handler on a borglet.
   Ex: (do :shell {:command \"ls ./\" :dir \"/home/\"})"
  [one & args]
  (if (keyword? one)
    (apply internal-do *client* one (take 1 args))
    (apply internal-do one (take 2 args))))

(defn getm
  "Returns a map of the specified keys from a borglet.
   Ex: (getm [:a :b]) => {:a 1 :b 2}"
  ([properties]
     (apply getm *client* properties))
  ([client properties]
     (do client :getm {:keys properties})))

(defn get
  "Returns the value of the specified key from a borglet.
   Ex: (get :a) => \"value of a\""
  ([property]
     (apply get *client* property))
  ([client property]
     (-> (getm client property)
         (property))))

(defn set
  "Takes a map of properties and sets those on a borglet."
  ([properties]
     (set *client* properties))
  ([client properties]
     (do client :set {:values properties})))

(defn delete
  "Deletes the specified properties from a borglet.
   Ex: (delete [:a :b])"
  ([properties]
     (apply delete *client* properties))
  ([client properties]
     (do client :delete {:keys properties})))