(ns borg.registry.core
  (:require [borg.registry.interface :as in]
            [borg.registry.s3 :as s3]
            [borg.transport.core :as transport]
            [clojure.string :as string])
  (:import [java.net NetworkInterface Inet4Address]))

(def registries {:s3 s3/init})
(def registry (atom nil))
(let [minute (* 60 1000)
      hour (* 60 minute)
      day (* 24 hour)]
  (def time-units {:minute minute
                   :hour hour
                   :day day}))

(defn cutoff-time [n unit]
  (* n (unit time-units)))

(defn ip
  "Returns a best guess at what the machine ip address is.
   This method may not work well if there are virtual interfaces present."
  []
  (->> (NetworkInterface/getNetworkInterfaces)
       (enumeration-seq)
       (map bean)
       (filter #(-> % :loopback not))
       (map :interfaceAddresses)
       first
       (map #(.getAddress %))
       (filter #(instance? Inet4Address %))
       first
       (.getHostAddress)))

(defn arg-parser [args]
  (let [f (first args)
        args (rest args)
        {ip :ip port :port classifiers :c}
        (if (string? f)
          {:ip f
           :port (first args)
           :c (rest args)}
          {:ip (ip)
           :port f
           :c args})]
    [ip port args]))

(defn set-registry!
  "Creates a new registry and sets it to be the global registry.
   Takes a map of config options that is passed to the registry init function."
  [registry config]
  (reset! registry ((registry registries) config)))

(defn register [& args]
  (apply in/register @registry (arg-parser args)))

(defn unregister [& args]
  (apply in/unregister @registry (arg-parser args)))

(defn get-borglet-names
  "Returns a list of borglet names in the form group:type:name-ip:port.
   It accepts an optional argument map,
   :cutoff => Exclude any borglets that haven't checked in since now
              minus :cutoff (in milliseconds), defaults to 30 minutes.
   :query => Exclude borglets that don't match the supplied name.
             Ex: {:query \"stage:web\"} would only return machines
             that are in the stage group and are identified as web machines
             You could also be less specific and only match by group {:query \"stage\"}.
   :expired? => Inverts :cutoff so that only borglets that haven't checked in inside
                the cutoff window are returned."
  [{:keys [cutoff query excluded?]}]
  (in/get-names @registry (or cutoff (cutoff-time 30 :minute)) query excluded?))

(defn purge-old-names [& [cutoff]]
  (in/purge-names @registry (or (cutoff (cutoff-time 30 :minute)))))

(defn name->connection [name]
  (-> (string/split name #"-")
      (last)
      (string/split #":")
      (update-in [1] #(Integer. %))
      (apply transport/client-create)))