;; owner: marshall@readyforzero.com
;; Prevents unlimited login attempts to a single user.
;; You can customize how many attempts are allowed before the
;; account is locked out, and for how long it remains locked.

(ns borg.auth.lockout
  (:require [borg.config :as config]
            [borg.internal.util :as util]
            [clojure.tools.logging :as lg])
  (:import java.util.Date))

(defonce tracker (atom nil))
(defonce tracker-settings (atom nil))

(defn set-settings!
  "Sets the lockout settings, takes a map of options.
   :max-attempts - The maximum number of login attempts before being locked out
   :lockout-duration - The number of minutes an account is locked out for."
  [opt]
  (swap! tracker-settings merge
         (if (:lockout-duration opt)
           (update-in opt [:lockout-duration] #(when % (util/unit->ms % :minute)))
           opt)))

(defn get-settings
  "If tracker-settings is nil, populates it with the values from
   borg.config, and returns those, otherwise returns tracker-settings."
  []
  (if-let [s @tracker-settings]
    (reset!
     tracker-settings
     {:max-attempts (config/get [:auth :max-attempts] 15)
      :lockout-duration (config/get [:auth :lockout-duration]
                                    5 #(util/unit->ms % :minute))})))

(defn unlock [identifier]
  (lg/info "unlocking identifier" identifier)
  (swap! tracker dissoc identifier)
  nil)

(defn lockout-expired?
  "Returns true if the last attempted login occured longer ago
   than the lockout-duration."
  [^Date last-attempt time-limit]
  (->> (.getTime last-attempt)
       (- (-> (Date.) .getTime))
       (< time-limit)))

(defn log-failure [identifier]
  (lg/info "Failed login attempt for" identifier)
  (swap! tracker update-in [identifier] #(-> (assoc % :time (Date.))
                                             (update-in [:count] (fnil inc 0)))))

(defn can-try?
  "Returns true if the if number of consecutive failed logins is less than
   the :max-attempts limit, or if the limit has been reached and the last login attempt
   was longer ago than the :lockout-duration"
  [identifier settings]
  (if-let [last-try (get @tracker identifier)]
    (if (>= (:count last-try) (:max-attempts settings))
      (when (and (:lockout-duration settings)
                 (lockout-expired? (:time last-try)
                                   (:lockout-duration settings)))
        (unlock identifier)
        true)
      true)
    true))

(defmacro auth-attempt
  "Wraps an authentication attempt and only tries to validate if the user
   is not locked out.
   Data should be the same map that is passed to the authenticator."
  [data & body]
  `(let [identifier# (:identifier ~data)]
     (if (can-try? identifier# @tracker-settings)
       (if-let [re# (do ~@body)]
         (do (log-failure identifier#)
             re#)
         (unlock identifier#))
       "You have tried to login too many times unsuccesfully, please try again later.")))
