;; owner: marshall@readyforzero.com
;; Authenticate using ssh keys.

(ns borg.auth.ssh
  (:require [aws.sdk.s3 :as s3]
            [borg.auth.interface :refer :all]
            [borg.aws.core :as aws]
            [clojure.java.io :as io])
  (:import [com.sshtools.j2ssh.transport.publickey
            SshPublicKeyFile SshPrivateKeyFile
            SshPublicKey SshPrivateKey]
           [com.sshtools.j2ssh.util Base64]))

;;;;;;;;;;;; Keystore ;;;;;;;;;;;;;;;
(defprotocol IKeystore
  (get-key [_ identifier])
  (add [_ identifier key])
  (change [_ identifier key])
  (delete [_ identifier]))

(defrecord S3Keystore [bucket]
  IKeystore
  (get-key [_ identifier]
    (let [obj (s3/get-object
                (aws/get-credentials)
                (:bucket _)
                identifier)]
      (with-open [content (:content obj)]
        (let [bytes (byte-array (-> obj :metadata :content-length))]
          (.read content bytes)
          (-> (SshPublicKeyFile/parse bytes)
              (.toPublicKey))))))

  (add [_ identifier path]
    (s3/put-object (aws/get-credentials)
                   (:bucket _)
                   identifier
                   (io/file path)))

  (change [_ identifier path]
    (add _ identifier path))

  (delete [_ identifier]
    (s3/delete-object (aws/get-credentials)
                      (:bucket _)
                      identifier)))

;;;;;;;;;;; Authenticator ;;;;;;;;;;;;;;
(defn load-private-key
  [path]
  (-> (io/file path)
      (SshPrivateKeyFile/parse)
      (.toPrivateKey nil)))

(defn load-public-key [path]
  (-> (io/file path)
      (SshPublicKeyFile/parse)
      (.toPublicKey)))

(defn encode-public-key
  "Takes a public key (from load-public-key)
   and encodes it into a json safe string."
  [^SshPublicKey key]
  (-> (.getEncoded key)
      (Base64/encodeBytes true)))


(defn sign [^SshPrivateKey private-key ^String content]
  (->> (.getBytes content)
       (.generateSignature private-key)
       (#(Base64/encodeBytes % true))))

(defn verify [^SshPublicKey public-key ^String signature ^String content]
  (.verifySignature public-key
                    (Base64/decode signature)
                    (.getBytes content)))

(defn internal-authenticate [user
                             ^SshPrivateKey private-key
                             ^SshPublicKey public-key]
  (let [encoded (encode-public-key public-key)]
    {:identifier user
     :data {:public-key encoded
            :signature (sign private-key encoded)}}))

(defn internal-validate
  "Takes a map of user names to encoded keys and a map
   created by (authenticate) and returns nil if successfull
   or an error message."
  [keystore data]
  (let [username (:identifier data)
        data (:data data)
        our-pub-key (get-key keystore username)]
    (cond
     (nil? username)
     "User can not be nil"
     (nil? our-pub-key)
     (str username " has not been added to this borglet.")
     (not= (encode-public-key our-pub-key) (:public-key data))
     (str "The public key does not match the one on file for " username)
     (not (verify our-pub-key (:signature data) (:public-key data)))
     "Could not verify signed message.")))

(defrecord SSH [keystore user]
  IAuth
  (add-user [_ username key-path]
    (add (:keystore _) username key-path))
  (add-users [_ users]
    (map #(add-user _ (:username %) (:key-path %)) users))
  (modify-user [_ username key-path]
    (change (:keystore _) username key-path))
  (remove-user [_ username]
    (delete (:keystore _) username))
  (authenticate [_]
    (let [user (:user _)]
      (internal-authenticate (:user user)
                             (:private user)
                             (:public user))))
  (validate-attempt [_ data]
    (internal-validate (:keystore _) data)))

(defn init
  "Takes a map with keys :keystore (borglet only)
   and :user (client only). :user must be another map with keys
   :user - your username
   :private - the path to your private key
   :public - the path to your public key"
  [options]
  (SSH. (:keystore options)
        (let [u (:user options)]
          {:user (:user u)
           :private (-> (:private u)
                        (load-private-key))
           :public (-> (:public u)
                       (load-public-key))})))
