;; owner: marshall@readyforzero.com
;; Handlers for borg.config and functions for registering and
;; unregistering functions that listen for changes.

(ns borg.borglet.config
  (:require [borg.borglet.handler.core :as h]
            [borg.config :as config])
  (:import java.util.UUID))

(h/defhandler getm
  "Get multiple keys as a map.
   Ex: (do :getm {:keys [:a :b :c]}) -> {:a 0 :b 1 :c 2}"
  [options]
  (->> (:keys options)
       (map keyword)
       (select-keys @config/state)))

(h/defauthedhandler set
  "Set a key(s) to a value(s).
   Ex: (do :set {:values {:a 1 :b 2}})"
  [options]
  (config/set (:values options))
  true)

(h/defauthedhandler delete
  "Delete key(s).
   Ex: (do :delete {:keys [:a :b :c]})"
  [options]
  (->> (:keys options)
       (map keyword)
       (config/delete))
  true)

(defn changed? [k old new]
  (not= (k old) (k new)))

(defn conditional-call
  "Only call func if listen-key is nil, or if
   listen-key is not equal in old and new."
  [old new func listen-key]
  (if listen-key
    (if (changed? listen-key old new)
      (func (listen-key old) (listen-key new)))
    (func old new)))

(defn add-listener
  "Adds a listener function to state, optionally
   listening to a specific key.
   Returns a key that can be used to remove the listener."
  [func & [listen-key]]
  (let [key (keyword (str (UUID/randomUUID)))]
    (add-watch config/state key
               (fn [k r o n]
                 (conditional-call o n func listen-key)))
    key))

(defn remove-listener
  "Removes a listener by the key returned from
   add-listener."
  [key]
  (remove-watch config/state key))