(ns borg.borglet.handler.core
  (:require [borg.auth.core :as auth]
            [borg.config :as config]
            [borg.borglet.users.core :as users]))

(def handlers (atom {}))
(def ^:dynamic *user* nil)

;;****************************
;; create an error or success response
;;****************************

(defrecord Response [status details error])

(defn make-response [success & [details error]]
  (->Response success details error))

(defn super-required []
  (make-response :super-required))

(defn auth-required []
  (make-response :auth-required))

(defn auth-fail [error]
  (make-response :auth-failed nil error))

(defn auth-success []
  (make-response :auth-success))

(defn make-error
  [msg & [details]]
  (make-response :fail details msg))

(defn make-success [details]
  (make-response :ok details))

(defn result->map [r]
  (select-keys r (keys r)))

(defn ->result [obj]
  (-> (if (= Response (type obj))
        obj
        (make-success obj))
      (result->map)))

(defmacro make-handler [handle restriction & args]
  (let [{all :a doc :d}
        (if (-> args first string?)
          {:d (first args) :a (rest args)}
          {:d nil :a args})
        params (first all)
        body (rest all)]
    `(if (and (config/get [:handlers :strict])
              (contains? @handlers ~(str handle)))
       (throw (Exception. ~(str "A handler with the name " handle " already exists")))
       (swap! handlers assoc ~(str handle)
              {:fn (fn ~params ~@body)
               :doc ~doc
               :restriction ~restriction}))))

;;*****************************
;; register and call handlers
;;*****************************

(defmacro defhandler
  "Creates a handler function.
   Params must be a vector of one arg, whose value will be a map."
  [handle & body]
  `(make-handler ~handle :none ~@body))

(defmacro defauthedhandler
  "Same as defhandler except the client must be authenticated."
  [handle & body]
  `(make-handler ~handle :auth ~@body))

(defmacro defsuperhandler
  "Same as defauthedhandler except the client user must be a super user."
  [handle & body]
  `(make-handler ~handle :super ~@body))

(defn call-handler
  "Ex: (call-handler {:handler \"shell\" :options {:command \"ls ~/\"})"
  [{:keys [handler options]} user authed]
  (binding [*user* (or user (config/get :user))]
  (-> (try
        (if-let [handler-map (get @handlers handler)]
          (let [res (:restriction handler-map)]
            (cond
             (and (or (= res :auth) (= res :super))
                  (not authed))
             (auth-required)
             (not (users/can-run? user res))
             (make-response
               :permission-error nil
               (str "You are not authorized to run handlers at the " res " level."))
             :else
             ((:fn handler-map) options)))
          (make-error (str "There is no handler with the name " handler)))
        (catch Exception e (do (doall (map println (.getStackTrace e)))
                               (make-error (.getMessage e)))))
      (->result))))

(defhandler handlers
  "Returns a map of the handlers with their
   doc string and if they require authentication."
  [options]
  (let [h @handlers]
    (->> (vals h)
         (map (juxt :restriction :doc))
         (zipmap (->> (keys h)
                      (map keyword))))))

(defhandler authenticate [options]
  (if-let [re (auth/validate-attempt options)]
    (auth-fail re)
    (-> (auth-success)
        (merge {:user (:user options)
                :authenticated true}))))

(defhandler logout [options]
  (-> (make-success true)
      (assoc :authenticated false)))

(defhandler sync-users [o]
  (users/sync-users))