(ns borg.borglet.util.io
  (:require [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [clojure.tools.logging :as lg])
  (:import [java.io File]))

(defn sh [user cmd & [opts]]
  (lg/info "SH" user cmd)
  (->> (into [] opts)
       (apply concat [cmd])
       (apply sh/sh "su" user "-c")))

(defn git-clone [user repo-url destination-dir ssh-key]
  (lg/info "Cloning repo" repo-url "to" destination-dir)
  (lg/info "Cloning as user" user "with key" ssh-key)
  (let [command (str "git clone " repo-url " " destination-dir)
        ssh-path (-> (io/resource "wrap-ssh4git.sh") io/file .getAbsolutePath)
        re (->>  (if ssh-key
                   (str "ssh-agent bash -c 'ssh-add " ssh-key ";
                         GIT_SSH="ssh-path " " command "'")
                   command)
                 (sh user))]
    (when (not= 0 (:exit re))
      (throw (Exception. (str "git-clone: " (:err re)))))))

(defn git-checkout [user repo-dir commit]
  (let [re (sh user (str "git checkout " commit) {:dir repo-dir})]
    (when (not= 0 (:exit re))
      (throw (Exception. (str "git-checkout: " (:err re)))))))

(defn git-revision [user]
  (-> (sh user "git rev-parse HEAD")
      (:out)))

(defn git-deploy-revision [{:keys [repo commit user ssh-key revisions-dir]}]
  (let [commit-dir (str revisions-dir "/" commit)]
    (lg/info "Deploying revision" commit "from" repo "to" commit-dir)
    (if (->> (File. commit-dir)
             (.exists)
             (not))
      (do
        (git-clone user repo commit-dir ssh-key)
        (git-checkout user commit-dir commit))
      (lg/info "Revision already exists"))))
