(ns borg.client
  (:require [borg.auth.core :as auth]
            [borg.registry.core :as registry]
            [borg.transport.core :as transport])
  (:refer-clojure :exclude [get set list]))

(def ^:dynamic *client* nil)

(defn authenticate
  "Authenticate with the client using the current authenticator.
   Throws an exception if authentication failed."
  [client]
  (let [re (transport/run-handler client :authenticate (auth/authenticate))]
    (when (not= "auth-success" (:status re))
      (throw (Exception. (str (:status re) " : " (:error re)))))))

(defn internal-run
  "Runs a handler. If the client returns an 'auth-required' message
   it will attempt to authenticate and then call the handler again.
   Throws an exception if there were errors."
  [client handler & [options]]
  (let [re (transport/run-handler client handler options)
        status (:status re)]
    (cond
      (= "auth-required" status)
      (do (authenticate client)
          (internal-run client handler options))
      (= "ok" status)
      (:details re)
      :else
      (throw (Exception. (str status " : " (:error re)))))))

(defn set-transport! [trans-key & [options]]
  (transport/set-transport! trans-key options))

(defn connect [host port]
  (transport/client-create host port))

(defn disconnect
  ([]
   (disconnect *client*))
  ([client]
    (transport/client-close client)))

(defmacro with-client [client & body]
  `(binding [*client* ~client]
     ~@body))

(defmacro with-clients
  "Perform a series of actions on multiple clients and return
   a list of their results."
  [clients & body]
  `(pmap
    #(try (with-client % ~@body)
          (catch Exception e# e#)) ~clients))

(defn list
  "Gets a list of borglet connections using the
   classifiers query, and an optional checkin cutoff"
  [& [classifiers cutoff]]
  (->> (registry/get-borglet-names {:query classifiers
                                    :cutoff cutoff})
       (map registry/name->connection)))

(defn run
  "Run a handler on a borglet.
   Ex: (run :shell {:command \"ls ./\" :dir \"/home/\"})"
  [one & args]
  (if (keyword? one)
    (apply internal-run *client* one (take 1 args))
    (apply internal-run one (take 2 args))))

(defn getm
  "Returns a map of the specified keys from a borglet.
   Ex: (getm [:a :b]) => {:a 1 :b 2}"
  ([properties]
     (apply getm *client* properties))
  ([client properties]
     (internal-run client :getm {:keys properties})))

(defn get
  "Returns the value of the specified key from a borglet.
   Ex: (get :a) => \"value of a\""
  ([property]
     (apply get *client* property))
  ([client property]
     (-> (getm client [property])
         (property))))

(defn set
  "Takes a map of properties and sets those on a borglet."
  ([properties]
     (set *client* properties))
  ([client properties]
     (internal-run client :set {:values properties})))

(defn deletem
  "Deletes multiple properties from a borglet.
   Ex: (deletem [:a :b])"
  ([properties]
     (apply deletem *client* properties))
  ([client properties]
     (internal-run client :delete {:keys properties})))

(defn delete
  "Delete a single property from a borglet.
   Ex (delete :a)"
  ([property] (delete *client* property))
  ([client property]
     (deletem client [property])))
