;; owner: marshall@readyforzero.com
;; A map of properties that is loaded from json file on disk.
;; Every time a the properties are changed it is saved back to disk.
;; Various modules get their configuration from here (transport, registry...)

(ns borg.config
  (:refer-clojure :exclude [get set])
  (:require [cheshire.core :as json]
            [clojure.java.io :as io]
            [clojure.tools.logging :as lg]))

(defonce config-location nil)
(defonce state (ref {}))

(defn init-map [m]
  (lg/info "Initialize config with map" m)
  (alter-var-root (var config-location) (constantly nil))
  (dosync (ref-set state m)))

(defn init-path [p]
  (lg/info "Initialize config with path" p)
  (alter-var-root
   (var config-location)
   (constantly (or p "/etc/borg/config.json")))
  (dosync
   (let [f (io/file config-location)]
     (->> (if (.exists f)
            (-> (slurp config-location)
                (json/parse-string true))
            {})
          (ref-set state)))))

(defn init [& [config]]
  (if (map? config)
    (init-map config)
    (init-path config)))

(defn save [map]
  (when config-location
    (->> (json/generate-string map {:pretty true})
         (spit config-location))))

(defn set
  ([k v]
     (dosync
      (-> (alter state assoc-in (if (keyword? k) [k] k) v)
          (save))))
  ([map]
     (dosync
      (-> (alter state merge map)
          (save)))))

(defmacro get
  "Gets the value from a nested structure using a either a single keyword
   or a sequence of keys.
   You can specify a default value in the event it would have returned nil,
   as well as a function the value will be passed through before being returned
   (both the real value and default value will be passed through the supplied fn).

   There is an optional fourth paramater that when set to true, the value found
   in state is nil, and not-found is not nil, it will set the value of path
   to not-found."
  [path & [not-found prep-fn set?]]
  `(let [prep-fn# (or ~prep-fn identity)
         value# (get-in @state (if (keyword? ~path) [~path] ~path))]
    (-> (if (nil? value#)
          (let [nf# ~not-found]
            (if (and nf# ~set?)
              (do (set ~path nf#) nf#)) nf#)
          value#) 
        (prep-fn#))))

(defn delete [keys]
  (dosync
   (-> (apply alter state dissoc)
       (save))))
