(ns borg.internal.lang)

(defmacro -?>
  "Equivalent to ->, except an intermediate falsy value stops threading the value through and returns nil."
  ([a] `(when-let [a-val# ~a] a-val#))
  ([a b] `(when-let [a-val# ~a] (-> a-val# ~b)))
  ([a b & cs] `(when-let [a-val# ~a] (-?> (-> a-val# ~b) ~@cs))))

(defmacro -?>>
  "Equivalent to ->>, except an intermediate falsy value stops threading the value through and returns nil."
  ([a] `(when-let [a-val# ~a] a-val#))
  ([a b] `(when-let [a-val# ~a] (->> a-val# ~b)))
  ([a b & cs] `(when-let [a-val# ~a] (-?>> (->> a-val# ~b) ~@cs))))

(defn maybe-throw-error [a-str a-val]
  (when (nil? a-val)
    (throw
     (Exception.
      (format "%s is nil, expecting non-nil." a-str)))))

(defn stringify
  "Returns a string, writing out 'nil' if nil."
  [x]
  (if (nil? x)
    "nil"
    (str x)))

(defmacro -!>
  "Equivalent to ->, except an intermediate nil value triggers an exception."
  ([a]
     (let [a-str (stringify a)]
       `(let [a-val# ~a]
          (maybe-throw-error ~a-str a-val#)
          a-val#)))
  ([a b]
     (let [a-str (stringify a)
           b-str (stringify b)
           exp-str (format "(-!> %s %s)" a-str b-str)]
       `(let [a-val# ~a]
          (maybe-throw-error ~a-str a-val#)
          (let [result# (-> a-val# ~b)]
            (maybe-throw-error ~exp-str result#)
            result#)
          )))
  ([a b & cs] `(-!> (-!> ~a ~b) ~@cs)))

(defmacro -!>>
  "Equivalent to ->>, except an intermediate nil value triggers an
   exception, and single arguments are allowed."
  ([a] `(-!> ~a))
  ([a b] (let [a-str (stringify a)
               b-str (stringify b)
               exp-str (format "(-!>> %s %s)" a-str b-str)]
           `(let [a-val# ~a]
              (maybe-throw-error ~a-str a-val#)
              (let [result# (->> a-val# ~b)]
                (maybe-throw-error ~exp-str result#)
                result#))))
  ([a b & cs] `(-!>> (-!>> ~a ~b) ~@cs)))