;; owner: marshall@readyforzero.com
;; Registry is the mechanism for borglets to announce themselves
;; and for clients to query them.
;;
;; The main method of identifying borglets and querying them are
;; classifiers.
;; You can think of classifiers as labels that go from
;; generic -> specific.
;;
;; For example, a beta machine that is serving your website using
;; production data might have the classifiers [:prod :web :beta].
;; If you wanted to get a list of all of your production web machines,
;; you would query with the classifiers [:prod :web] and get back
;; (\"prod:web:web1:<ip>:<port>\" \"prod:web:beta:<ip>:<port>\" ...)"
;;
;; -- Config --
;; Key :registry
;; Sub-keys
;; :type - (string) registry type (ex: s3, virtual), default virtual.
;; :options - (map) options that will be passed to the registry init
;;            function.
;; :interval - (optional) the interval (in minutes) that the borglet will check
;;             in, default 30 minutes.
;; :classifiers - (optional) list of classifier strings.

(ns borg.registry.core
  (:require [borg.internal.util :as util]
            [borg.internal.module :as m]
            [borg.registry.interface :as in]
            [borg.registry.s3 :as s3]
            [borg.registry.virtual :as virtual]
            [borg.transport.core :as transport]
            [clojure.string :as string])
  (:import [java.net NetworkInterface Inet4Address]))

;; see borg.internal.module for behavior
(m/create-vars {:s3 s3/init
                :virtual virtual/init})

(defn add-registry-type! [key fn]
  (m/add-type key fn))

(defn get-registry []
  (m/get-instance :registry :virtual))

(defn set-registry! [reg & [config]]
  (m/set-instance reg config))
;; end module section

(defn ip
  "Returns a best guess at what the machine ip address is.
   This method may not work well if there are virtual interfaces present,
   in which case you should provide the ip when registering a borglet."
  []
  (->> (NetworkInterface/getNetworkInterfaces)
       (enumeration-seq)
       (map bean)
       (filter #(-> % :loopback not))
       (map :interfaceAddresses)
       first
       (map #(.getAddress %))
       (filter #(instance? Inet4Address %))
       first
       (.getHostAddress)))


(defn register
  "Register a borglet. Takes an ip address as the first
   argument, then a port and a list of classifiers.
   Ex: (register \"10.0.10.106\" 2323 [:stage :daemon])
       (register 2323 [:prod :web :beta])"
  [ip port classifiers]
  (in/register (get-registry) ip port classifiers))

(defn unregister
  "Unregister a borglet. Takes the exact same arguments as register."
  [ip port classifiers]
  (in/unregister (get-registry) ip port classifiers))

(defn get-borglet-names
  "Returns a list of borglet names in the form group:type:name-ip:port.
   It accepts an optional argument map,
   :cutoff => Exclude any borglets that haven't checked in since now
              minus :cutoff (in milliseconds), defaults to 30 minutes.
   :query => Exclude borglets that don't match the supplied classifiers
             Ex: {:query [:stage :web]} would only return machines
             that start with the classifier \"stage:web\"
             You could also be less specific and only match by
             the first classifier {:query [:stage]}.
   :expired? => Inverts :cutoff so that only borglets that haven't checked
                in inside the cutoff window are returned."
  [{:keys [cutoff query excluded?]}]
  (in/get-names (get-registry) (or cutoff (util/unit->ms 30 :minute)) query excluded?))

(defn purge-old-names
  "Remove borglet names from the registry that haven't
   checked in within the cutoff window. Defaults to 30 minutes."
  [& [cutoff]]
  (in/purge-names (get-registry) (or cutoff (util/unit->ms 30 :minute))))

(defn name->connection
  "Converts a borglet name into a connection."
  [name]
  (-> (string/split name #":")
      (->> (take-last 2)
           (into []))
      (update-in [1] #(Integer. %))
      (->> (apply transport/client-create))))
