;; Owner: wolfson@readyforzero.com
;; Client entry point (set-state!) and borglet entry point
;; (state-update).

;; High-level overview:

;; - The client defines a graph describing the state to be applied.
;;   The nodes in of the graph are maps with the structure
;;
;;      {:provides :name-of-node
;;       :requires list-of-dependencies
;;       :produces-ops list-of-ops
;;       :attrs {:type "type of node", :attr1 value1, :attr2 value2, ... }}
;;
;;   At a minimum, each node in the graph has a *type* corresponding
;;   to the kind of resource (file, package, user, etc.) the node
;;   represents, a *name* that can be used to refer to it (in
;;   particular, by other nodes), and a (possibly empty)
;;   *requirements* list, containing the names of the nodes the node
;;   under consideration depends on. Several predefined node types are
;;   implemented in borg.state.types.provided.

;; - The client serializes the graph and sends it to a borglet (or many
;;   borglets).

;; - A borglet receives the graph, and deserializes it. The borglet
;;   then traverse the graph in topological order. For each node, the
;;   borglet obtains a list of actions it must take in order to make
;;   the machine's actual state conform to the state the node
;;   describes.

;; - The borglet then executes each of the actions in order. If any
;;   action fails, no further actions are taken.

;; - Regardless, a description of actions taken is then returned to the
;;   client. In the event of failure, an error message is also returned,
;;   and the action that failed is indicated.


(ns borg.state.core
  (:require [borg.borglet.handler.core :as handler]
            [borg.client :as client]
            [borg.state.graph :as g]
            [borg.state.internal.core :as i]
            [borg.state.types.drive]
            [borg.state.types.fs]
            [borg.state.types.package]
            [borg.state.types.process]
            [borg.state.types.repo]
            [borg.state.types.users]))

(defn run! [state dry-run?]
  (let [node-lists (g/graph-from-wire state)
        action-map-lists (g/check-nodes node-lists)]
    (i/run-action-lists action-map-lists dry-run?)))

(handler/defauthedhandler state-update
  "Update the state of the machine according to the description in the
   state graph. Returns a description of actions taken.

   If the dry-run? option is true, only returns a description of the
   actions that would be taken, without actually performing them."
  [{:keys [state dry-run?]}]
  (run! state dry-run?))

(defn set-state!
  "Set the state on the current connection."
  [state dry-run?]
  (client/run :state-update {:state (g/graph-to-wire state) :dry-run? dry-run?}))

(defn merge-graphs
  "Merge two or more graphs together.

   Necessary if nodes of one graph specify their dependencies by name:

      (def d (directory ... {:provides :some-directory ...}))
      (def f (file ... {:provides :some-file :requires [:some-directory]}))

      (def complete-graph (merge-graphs f d))

  The merge-graphs call is necessary to connect f's reference
  to :some-directory up with d.

  It would not be necessary if f referred directly to d:

     (def f (file ... {:provides :some-file :requires [d]}))."
  [& args]
  (apply g/merge-graphs args))
