;; Owner: wolfson@readyforzero.com

;; Utilities for implementing operations on nodes. Functions and
;; macros are provided for:

;; handling errors (try-to and sh-result).

;; defining functions that describe their behavior (defop)---these are
;; referred to as "actions".

;; defining functions that describe how to go from state nodes to
;; lists of actions (path and node-path).

;; the minimum required to provide an implementation for a particular
;; state node type is an implementation of
;; borg.state.graph/check-node, which should return a list of actions.
;; (See package.clj for a minimal example.)

;; Each action, when executed, should return a "status" map, with a
;; :status key whose value is either :ok or :error. If it is :error,
;; it should have an additional key, :reason, with an error message
;; (or error value, it can be any JSON-serializable value). If the
;; status is :ok, the map may also contain a :log key.

(ns borg.state.types.core
  (:require [borg.state.types.internal.core :as i]
            [borg.state.util :as u]
            [clojure.core.match :as m]
            [clojure.string :as str]
            [macroparser.bindings :as b]
            [macroparser.functions :as f]
            [the.parsatron :as parsatron]))

;; Handling errors
(defmacro try-to
  "Return an \"ok\" status if forms execute without raising an
  exception. If an exception is raised, return an error status with an
  error message from \"msg\" and the message in the exception
  object."
  [msg & forms]
  `(try
     ~@forms
     (u/ok)
     (catch Exception e#
       (u/error (str "Exception trying to " ~msg ": " (.getMessage e#))))))

(defn sh-result
  "Return an \"ok\" status if the exit key of the argument is 0. The
  \"out\" key of the argument will be included as a log.

   Otherwise, return an error status with the entire result object as
   the error message."
  
  [{:keys [exit out err] :as result}]
  (if (== exit 0)
    (u/ok out)
    (u/error result)))

;; Macro for creating functions that auto-destructure nodes

(defmacro defnodefn
  "Like defn, but creates two functions: one which is identical to
  what would be created with defn, and one in which the arguments are
  supplied by destructing the attribute map of a node. The second
  function has \"node-\" prefixed to its name. E.g.:

    (defnodefn exists? [pathname] (fs/exists? pathname))

  expands to

    (do (defn exists? [pathname] (fs/exists? pathname))
        (defn node-exists? {{:keys [pathname]} :attrs} (fs/exists? pathname))

  Only one arity is allowed."
  [& fn-args]
  (let [parsed (parsatron/run (f/parse-defn-like) fn-args)
        name (:name parsed)]
    (assert (== 1 (count (:arities parsed))) "Only one arity allowed.")
    (let [arity (first (:arities parsed))
          bindings (-> arity :params b/bound-symbols)
          rebound (assoc arity :params (i/replacement-bindings (gensym) bindings))]
      `(do ~(f/unparse-defn-like parsed)
           ~(f/unparse-defn-like (assoc parsed :arities [rebound] :name (symbol (str "node-" name))))))))

;; Macro for creating functions that return a representation of what they do

(defmacro defop
  "Create a function that, when called, returns a representation of
   the action to be performed, along with a nullary function that
   actually performs it. This is implemented with defnodefn, so it
   *also* creates a function that destructures a node's attribute
   map. E.g.:

     (defop write [pathname contents] (spit pathname contents))

   expands to

     (defnodefn write [pathname contents]
         {:args (zipmap [:pathname :contents] [pathname contents])
          :body '(fn [] (spit pathname contents))
          :fn (fn [] (spit pathname contents))
          :op :write})

   and ultimately creates functions named write and node-write."

  [& fn-args]
  (let [parsed (parsatron/run (f/parse-defn-like) fn-args)
        name (:name parsed)]
    (assert (== 1 (count (:arities parsed))) "Only one arity allowed!")
    (let [arity (i/update-arity i/update-body-op (:name parsed) (first (:arities parsed)))]
      (f/unparse-defn-like (assoc parsed :arities [arity] :type `defnodefn)))))

(defn chain-op
  "Return an op that runs op1, followed by op2 if op1 succeeded.

   If op1 was passed different arguments than op2, but those arguments
   had the same name, then the :args map in the op that this returns
   will be deceptive. The individual ops, however, will run with their
   correct args."
  [op1 op2]
  (let [args (merge (:args op1) (:args op2))
        opname (keyword (str (name (:op op1)) "-then-" (name (:op op2))))
        body (str "(fn [] (chain-status (" (:body op1) ") (" (:body op2) ")))")
        fn (fn [] (u/chain-status ((:fn op1)) ((:fn op2))))]
    {:args args
     :op opname
     :body body
     :fn fn}))

;; Macros for describing what actions to take for a given node. (the
;; main work is done in path, but the docstring is in node-path since
;; it is expected that it will be used more frequently.)

(defmacro path
  [& path-elems]
  (let [parsed (->> (parsatron/run (i/parse-path-elems) path-elems)
                    (map i/trans-splices)
                    (map (fn [elem]
                           (if (and (= :plain (:type elem))
                                    (i/is-pred? (:pred elem)))
                             {:type :explicit :pred (:pred elem) :op (i/un-pred (:pred elem))}
                             elem)))
                    (mapv (fn [elem]
                            (case (:type elem)
                              :plain (i/unparse-plain (:pred elem))
                              :explicit (let [argsym (gensym "arg")]
                                          `(fn [~argsym] (when ~(i/unparse-op (:pred elem) argsym)
                                                          [~(i/unparse-op (:op elem) argsym)])))
                              :splice `(fn [arg#] ~(:lst elem))))))]
    `(fn [node#] (apply concat (map #(% node#) ~parsed)))))

(defmacro node-path
  "A mini-language for specifying how to select what actions to
   perform, given a state node. Expands into a function that expects a
   state node as its argument and returns a list.

   Elements of the mini-language are either:

      (a) a symbol

      (b) a form

      (c) symbol-or-form => symbol-or-form

   with two special cases:

      1. a single symbol ending with a question mark:

        symbol?

      is equivalent to:

        symbol? => symbol

      2. A form whose first element is
      clojure.core/unquote-splicing (i.e. what the reader produces on
      seeing ~@foo) is assumed to have a second element that evaluates
      to a seqable, and that element is spliced into the result of
      calling the function node-path creates.

   Bare symbols in either (a) or (c) are prefixed with \"node-\" if
   they don't already begin with \"node-\". (To avoid this, just use
   the \"path\" macro, which does no prefixing.) The semantics
   for (a)--(c) are:

      (a) the function denoted by the symbol is called on the argument node.

      (b) the argument node is ignored and the result of evaluating
          the form is returned.
   
      (c) if evaluation according to the rules just given of the
          symbol-or-form preceding the => is truthy, evaluate the
          symbol-or-form after the => according to the rules just
          given.

   Example:

      (node-path chown?
                 should-chmod? => (do-chmod pathname)
                 ~@list-of-actions
                 (and test1 test2) => (write file contents)

   Evaluates to a function that takes a state node and:

      calls node-chown on the argument node if node-chown? returns truthy when
      called on the argument node,

      evaluates (do-chmod pathname) if node-should-chmod? returns
      truthy when called on the argument node,

      splices the value of list-of-actions into the result,

      evaluates (write file contents) if (and test1 test2) is truthy."
  [& path-elems]
  (let [prefixed (map i/prefix-node-sym path-elems)]
    `(path ~@prefixed)))
