(ns borg.state.types.internal.drive
  (:require [borg.internal.lang :refer [-!>>]]
            [clojure.java.shell :as sh]
            [clojure.string :as string]))

(let [k 1024
      m (* 1024 k)
      g (* 1024 m)]
  (def units {:b 1 :kb k :mb m :gb g}))

;; Drive partition types [<string code> <id>]
(def types {:normal ["L" 83]
            :swap ["S" 82]
            :extended ["E" 5]
            :linux-extended ["X" 85]})

(defn get-unit [u]
  (if (keyword? u)
    ((-> u name string/lower-case keyword) units)
    u))

(defn unit->unit
  "Converts a value from one unit to the other. A unit can either be of
   of the keywords from \"units\", or a number."
  [val start-unit goal-unit]
  (-> (* val (get-unit start-unit))
      (/ (get-unit goal-unit))))

;; -- TO WIRE --
(defn update-size [spec]
  (-> (assoc spec :size (when-not (= :fill (:size spec))
                          (unit->unit (:size spec)
                                      (or (:unit spec) :mb)
                                      :mb)))
      (dissoc :unit)))

(defn update-type [spec]
  (->> (or (:type spec) :normal)
       (get types)
       (assoc spec :type)))

;; -- FROM WIRE --

;; Get drive info
(defn drive-id [id]
  (str "/dev/" id))

(defn lines->list [input]
  (string/split input #"\n"))

(defn find-unit [lines pattern]
  (->> (map #(re-find pattern %) lines)
       (keep identity)
       first
       last
       (Long.)))

(defn with-sectors
  "Adds keys :sector-size and :sectors to the drive-spec,
   where :sector-size is in bytes and :sectors is the total
   count of sectors the drive contains."
  [drive-specs]
  (let [lines (-!>> (:drive drive-specs)
                    (drive-id)
                    (sh/sh "fdisk" "-l")
                    :out
                    lines->list)
        s (find-unit lines  #"^Sector size .* (\d+) bytes$")]
    (-> (assoc drive-specs :sector-size s)
        (assoc :sectors (/ (find-unit lines #"^Disk .* (\d+) bytes$") s)))))

(defn parse-partition-line
  "Takes the output of sfdisk --dump and returns a list of maps with keys
   :start :size :type, where :start and :size are in terms of sectors,
   and :type is the numerical id of the partition type."
  [info]
  (->> (string/split info #",")
       (map #(->> (string/split % #"=")
                  (last)
                  (string/trim)))
       (take 3)
       (map #(Long. %))
       (zipmap [:start :size :type])))

(defn get-drive-info
  "Returns a list of maps with keys :start :size :type."
  [drive-identifier]
  (let [drive-id (drive-id drive-identifier)
        re (re-pattern (str "^" drive-id))]
    (-!>> drive-id
          (sh/sh "sfdisk" "--dump" "--unit" "S")
          (:out)
          lines->list
          (filter #(re-find re %))
          (map parse-partition-line))))

;; Convert partition specs

(defn with-start
  "For use with reductions, sets :start on cur to the sum of :start and :size from prev,
   or if prev is nil sets it to 1. Assumes sizes are defined in mb."
  [prev cur]
  (assoc cur :start (if prev (+ (:start prev) (:size prev)) 1)))

(defn mb->sector [v sector-size]
  (when v (unit->unit v :mb sector-size)))

(defn units->sectors
  "Convert :size and :start to sectors."
  [sector-size spec]
  (-> spec
      (update-in [:size] mb->sector sector-size)
      (update-in [:start] mb->sector sector-size)))

(defn prepare-specs
  "Add :start keys to all partitions, if :size is nil set it to the number of
   remaining unused sectors."
  [specs sector-size sector-count]
  (->> (reductions with-start nil specs)
       rest
       (map units->sectors (repeat sector-size))
       (map #(if-not (:size %)
               (assoc % :size (- sector-count (:start %)))
               %))))

(defn shift-partition
  "Shift the start of the partition by + diff.
   If :start + :size of the spec is equal to total-sectors
   subtract diff from :size."
  [total-sectors diff spec]
  (-> (if (= (+ (:size spec) (:start spec)) total-sectors)
        (update-in spec [:size] - diff)
        spec)
      (update-in [:start] + diff)))

(defn with-extended-partition
  "If the total number of partitions is greater than 4 insert an :extended
   partition as the 4th partition to allow for more than 4.
   This insertion requires moving the start of the partitions after extended by n
   where n is the partition number after the extended partition."
  [sector-count specs]
  (if (> (count specs) 4)
    (let [start (take 3 specs)
          s (apply + (-> start last (select-keys [:start :size]) vals))]
      (concat start
       [{:start s
         :size (- sector-count s)
         :type  (:extended types)}]
       (->> (drop 3 specs)
            (map shift-partition (repeat sector-count) (iterate inc 1)))))
    specs))

(defn partition->sfdisk [spec]
  (->> (map get (repeat spec) [:start :size :type])
       (string/join ",")))

(defn partitions->sfdisk [specs]
  (->> (map partition->sfdisk specs)
       (string/join \newline)))