;; Owner: wolfson@readyforzero.com
;; Implementation for user and group node types.
(ns borg.state.types.users
  (:require [borg.state.graph :as g]
            [borg.state.util :as u]
            [clojure.algo.generic.functor :as f]
            [clojure.core.match :as m]
            [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [clojure.string :as str]
            [me.raynes.fs :as fs])
  (:use [borg.state.types.core :only [defop sh-result defnodefn]]))

(defn read-passwd-like
  "Read a file consisting of lines with colon-separated fields. The
  \"format\" param is a list of field names as keywords; there should
  be as many keywords as there are fields. Returns a map indexed by
  the key-by parameter. If any value in the file is uninteresting it
  may be given a name of \":_\"."
  [format key-by file]
  (let [num-fields (count format)
        lines (line-seq (io/reader file))]
    (->> lines
         (map #(str/split % #":" num-fields))
         (map (fn [line]
                (->> line
                     (map (fn [col-name col-val]
                            (when-not (= col-name :_)
                              [col-name col-val]))
                          format)
                     (filter identity)
                     (into {}))))
         (group-by key-by)
         (f/fmap first))))

(defn split-when-nonempty [s pat]
  (if (empty? s)
    []
    (str/split s pat)))

(defn read-groups []
  (f/fmap #(update-in % [:users] split-when-nonempty #",")
          (read-passwd-like [:group-name :_ :gid :users] :group-name "/etc/group")))

(defn read-users []
  (read-passwd-like [:username :_ :uid :gid :_ :home :shell] :username "/etc/passwd"))

(defop mod-user-groups [name groups]
  (sh-result
   (sh/sh "usermod" "-G" (str/join "," groups) name)))

(defop create-user [name groups]
  (sh-result
   (if (seq groups)
     (sh/sh "useradd" "-U" "-m" "-G" (str/join "," groups) "-b" "/home" name)
     (sh/sh "useradd" "-U" "-m" "-b" "/home" name))))

(defn user-home [username]
  (str "/home/" username))

(defop delete-user [name]
  (sh-result (sh/sh "userdel" name)))

(defop shred-user-home [name]
  (u/chain-status
   (sh-result (sh/sh "find" (user-home name) "-type" "f" "-exec" "shred" "--remove" "{}" ";"))
   (if (fs/delete-dir (user-home name))
     (u/ok)
     (u/error (str "Could not delete " (user-home name))))))

(defnodefn check-user [name groups delete?]
  (let [pwd (-> (read-users) (get name))
        supplemental-groups (->> (read-groups)
                                 (keep (fn [[group-name {:keys [users]}]]
                                         (when (some #(= name %) users)
                                           group-name)))
                                 seq)]
    (m/match [pwd delete? (= groups supplemental-groups)]
             [nil true _ ]  nil
             [_ true _]     [(delete-user name) (shred-user-home name)]
             [nil _ _]      [(create-user name groups)]
             ;; since all you can do is specify supplemental groups,
             ;; we don't check the current state of the user.
             ;; NB this assumes that no one has changed the user.
             [_ _ true]     nil
             [_ _ false]    [(mod-user-groups name groups)])))

(defop create-group [name gid]
  (sh-result
   (sh/sh "groupadd" "-g" gid name)))

(defop mod-group [name gid]
  (sh-result
   (sh/sh "groupmod" "-g" gid name)))

(defnodefn check-group [name gid]
  (let [grp (-> (read-groups) (get name))
        old-gid (:gid grp)]
    (m/match [grp (= gid old-gid)]
             [nil _]    [(create-group name gid)]
             [_ true]   nil
             [_ false]  [(mod-group name gid)])))

(defmethod g/check-node "user" [node _ _]
  (node-check-user node))

(defmethod g/check-node "group" [node _ _]
  (node-check-group node))
