(ns borg.state.util
  (:require [clojure.core.match :as m]
            [clojure.set :as set]
            [clojure.string :as str]))

(def validators (atom {}))

(defmacro validator-maker
  "Create a validator.

   A validator is either a symbol, or a seq of two or three
   elements: (symbol validator-fn) or (symbol validator-fn transform-fm)

   If the validator is simply a symbol, it always validates its input
   and never transforms it.

   If the validator is a seq of (symbol validator-fn), the
   validator-fn should return a truthy value if its argument is
   correct, and falsy otherwise.

   If the validator is a seq of (symbol validator-fn transform-fn),
   the validator-fn should be as above and the transform-fn should
   return a canonical representation of its argument. The transform-fn
   can assume that its argument has passed the validator-fn. The
   result of the validation is the result of the transform-fn.

   E.g.:

   - (validator-maker anything) creates a validator that accepts everything

   - (validator-maker user (some-fn string? keyword?)) creates a
     validator that accepts only strings or keywords;

   - (validator-maker user (some-fn string? keyword?) name) creates a
     validator that accepts only strings or keywords and returns strings."
  [name]
  (if (seq? name)
    (let [[name validator transform] name]
      `(swap! validators assoc ~(keyword name)
              (fn [obj#]
                (if (~validator obj#)
                  (~(or transform identity) obj#)
                  (throw (Exception. (str "attribute " ~(str name) " failed validator for arg " obj#)))))))
    `(swap! validators assoc ~(keyword name) identity)))

(defmacro make-validators
  "Create several validators at once."
  [& keys]
  (if (seq keys)
    `(do (validator-maker ~(first keys))
         (make-validators ~@(rest keys)))
    nil))

(defn merge-attrs [acc attr]
  (if (contains? acc (first (keys attr)))
    (throw (Exception. "Duplicate attribute specified: " (key attr)))
    (merge acc attr)))

(defn string-or-keyword? [x] (or (string? x) (keyword? x)))

(defn dependencies [{:keys [requires]}]
  (let [{depend-names true depend-graphs false} (group-by keyword? requires)]
    [(concat depend-names (map (comp :provides :node) depend-graphs))
     (reduce merge {} (map :nodemap depend-graphs))]))

(validator-maker (provides keyword?))
(validator-maker requires)

(defn check-attrs [name attr-map required]
  (doseq [r required]
    (assert (contains? attr-map r) (str "Node type " name " requires attribute " r)))
  (reduce (fn [acc key]
            (assoc acc key ((get @validators key identity) (attr-map key))))
          {}
          (keys attr-map)))

(defmacro node-type
  "Create a function for creating nodes.

   Syntax: (node-type node-name :arguments [...] :required-attrs [...] :optional-attrs [...]
                :produces-ops [...])

   The function created will have as many positional arguments as
   there are elements in the :arguments list.

   The final argument is a map whose keys are keywords corresponding
   to the required-attrs and optional-attrs parameters. These are
   validated and transformed by the entries in the validators
   atom (see make-validator). A \":provided\" entry is always
   required, and a \":requires\" entry is always optional, regardless
   of whether or not this is specified.

   The :produces-ops list should enumerate the actions that can be
   returned from check-node when called on a node of this type. An
   exception will be raised if an unrecognized action is returned.

   The result of calling a function defined with node-type is a
   minimal graph containing only one node. The node has the form

   {:provides provides
    :requires requirements
    :attrs attr-map}

   where the name under :type is the name of the function.

   E.g., given

   (node-type directory :arguments [pathname] :required-attrs [user permissions]
              :produces-ops [create-dir chown])

   The call (directory \"/some/dir\" {:provides :some-dir :user \"root\" :permissions \"0777\"})

   results in the structure

   {:node {:provides :some-dir
           :requires ()
           :attrs {:pathname \"/some/dir\"
                   :user \"root\"
                   :type \"directory\"
                   :produces-ops [:create-dir :chown]
                   :permissions \"0777\"}}
    :nodemap {:some-dir {:provides :some-dir
                         :requires ()
                         :attrs {:pathname \"/some/dir\"
                                 :type \"directory\"
                                 :produces-ops [:create-dir :chown]
                                 :user \"root\"
                                 :permissions \"0777\"}}}}"
  [name & {:keys [required-attrs optional-attrs arguments produces-ops] :as options}]
  (let [required-attrs (mapv keyword required-attrs)
        optional-attrs (mapv keyword optional-attrs)
        produces-ops (mapv keyword produces-ops)
        required-attrs (if-not (some #(= :provides %) required-attrs)
                         (vec (cons :provides required-attrs))
                         required-attrs)
        attrs (concat required-attrs optional-attrs)
        attrs (if-not (some #(= :requires %) attrs)
                (cons :requires attrs)
                attrs)
        direct (map symbol arguments)
        direct-kw (map keyword direct)]
    (assert (and required-attrs arguments)
            "Both :arguments and :required-attrs arguments to node-type are required.")
    `(defn ~name [~@direct attr-map#]
       (let [dp-pairs# (map vector ~(vec direct-kw) ~(vec direct))
             attr-map# (check-attrs ~(str name) (merge (into {} dp-pairs#) attr-map#) ~required-attrs)
             attr-map# (assoc attr-map# :type ~(str name))
             [depend-names# depend-graphs#] (dependencies attr-map#)
             provides# (:provides attr-map#)
             attr-map# (dissoc attr-map# :requires :provides)]
         (when (some (partial = provides#)  depend-names#)
           (throw (Exception. (str provides# " cannot depend on what it provides!"))))
         (let [n# {:provides provides# :attrs attr-map# :requires depend-names#
                   :produces-ops ~produces-ops}]
           {:nodemap (merge {provides# n#} depend-graphs#) :node n#})))))

(defn ok? [res]
  (= (:status res) :ok))

(defn ok [& [log]]
  {:status :ok :log log})

(defn error [reason]
  {:status :error :reason reason})

(defn status [res attr]
  (when (ok? res)
    (get res attr)))

(defmacro defnmatch [name doc? & clauses]
  (let [[doc clauses] (if (string? doc?)
                        [doc? clauses]
                        ["" (cons doc? clauses)])]
    (assert (symbol? name) (str "Expected symbol, got " name))
    (assert (every? seq? clauses) (str "Clauses of match fn " name " must be seqs!"))
    (assert (== (count (set (map #(count (first %)) clauses))) 1)
            (str "Clauses of match fn " name " must have equal arity!"))
    (let [arity (count (ffirst clauses))
          syms (repeatedly arity gensym)
          patterns (map first clauses)
          actions (map rest clauses)
          clauses (interleave (mapv (comp vector vector) patterns) actions)]
      `(defn ~name ~doc [~@syms]
         (m/match [[~@syms]]
                  ~@(apply concat clauses))))))

;; basically error + writer monad
(defnmatch chain-status*
  ([({:status :ok} :as res1) then]
     (let [nextres (then)]
       (m/match [nextres]
                [{:status :ok} :as res2] {:status :ok :log (remove nil? (flatten [(:log res1) (:log res2)]))}
                [{:status :error :reason reason}] nextres)))
  ([({:status :error :reason reason} :as err) _] err))

(defmacro chain-status
  "Evaluate st1, which should return a map with a :status key whose
   value is either :ok or :error, and then evaluate st2 if the :status
   of st1 is :ok.

   Returns the status of st1 if it is :error, or the status of st2
   otherwise."
  [st1 st2]
  `(chain-status* ~st1 (fn [] ~st2)))

(defn transitive-dependencies [node nodemap]
  (let [my-requires (set (map nodemap (:requires node)))
        their-requires (map #(transitive-dependencies % nodemap) my-requires)]
    (reduce set/union my-requires their-requires)))
