;; Owner: wolfson@readyforzero.com
;; Pretty pictures of state graphs using graphviz.
;; Note that visualize-graph requires X and actually pops up an X window
(ns borg.state.viz
  (:require [borg.state.internal.viz :as i]
            [clojure.algo.generic.functor :as f]
            [clojure.java.shell :as sh]
            [clojure.string :as str]))

(defn run-dot [dot]
  (sh/sh "dot" "-Tx11" :in dot))

(defn visualize-ok-status [{:keys [shutdowns actions]} graph-name]
  (let [nodemaps (i/make-action-node-map actions)
        shutdown-cluster (->> shutdowns
                              (mapcat vals)
                              i/mk-op-records)
        nodemaps (if (empty? shutdowns)
                   nodemaps
                   (cons {"shutdowns" shutdown-cluster} nodemaps))
        clusters (i/make-clusters nodemaps)
        graph (i/make-graph graph-name [clusters])]
    (spit (str "/tmp/" graph-name ".dot") graph)
    (run-dot graph)))

(defn visualize-shutdown-error-status [{:keys [shutdowns planned]} graph-name]
  (let [nodemaps (i/make-action-node-map planned)
        shutdown-map (i/make-error-maps {:shutdowns (-> shutdowns
                                                      (update-in [:planned] (partial mapcat vals))
                                                      (update-in [:executed] (partial mapcat vals)))})
        shutdown-cluster (i/make-clusters [shutdown-map] "shutdown" "red")
        node-clusters (i/make-clusters nodemaps)
        graph (i/make-graph graph-name [shutdown-cluster node-clusters])]
    (spit (str "/tmp/" graph-name ".dot") graph)
    (run-dot graph)))

(defn visualize-action-error-status [{:keys [shutdowns errors oks executed planned]} graph-name]
  (let [executed-maps (i/make-action-node-map executed)
        ok-maps (first (i/make-action-node-map [(f/fmap :actions oks)]))
        error-maps (i/make-error-maps errors)
        not-reached-maps (i/make-action-node-map (drop (inc (count executed)) planned))
        shutdown-cluster (->> shutdowns (mapcat vals) i/mk-op-records)
        exec-c  (i/make-clusters (if (empty? shutdowns)
                                   executed-maps
                                   (cons {"shutdowns" shutdown-cluster} executed-maps)) "exec" "blue")
        failed-c (i/make-clusters [(merge ok-maps error-maps)] "failed" "red")
        unreached-c (i/make-clusters not-reached-maps "unreached")
        graph (i/make-graph graph-name [exec-c failed-c unreached-c])]
    (spit (str "/tmp/" graph-name ".dot") graph)
    (run-dot graph)))

(defn visualize-results-graph [g & [graph-name]]
  (let [dry-run? (:dry-run? g)
        graph-name (or graph-name "result_state_graph")]
    (if (= :ok (keyword (:status g)))
      (visualize-ok-status g graph-name)
      (if (= :actions (:error-location g))
        (visualize-action-error-status g graph-name)
        (visualize-shutdown-error-status g graph-name)))))

(defn visualize-local-graph [g & [graph-name]]
  (let [graph-name (or graph-name "state_graph")
        nodes (vals (:nodemap g))
        edges (->> nodes
                   (map (juxt :provides :requires))
                   i/flatten-edges)
        dot (i/fmt-dot edges graph-name)]
    (spit (str "/tmp/" graph-name ".dot") dot)
    (run-dot dot)))

(defn visualize-graph
  "Display the graph g, writing out the graph to /tmp/ using the
   supplied graph-name or, if graph-name is nil, \"state_graph\" (if
   it is a graph of state to be sent to a borglet) or
   \"result_state_graph\" if it is a response from a borglet."
  [g & [graph-name]]
  (if (contains? g :status)
    (visualize-results-graph g graph-name)
    (visualize-local-graph g graph-name)))
