(ns buckshot.scheduler
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util])
  (:import [java.util.concurrent ScheduledThreadPoolExecutor TimeUnit]))

(defprotocol IScheduler
  (add-job! [this job])
  (del-job! [this job])
  (start!   [this])
  (stop!    [this]))

(defn- add-job-fn [{:keys [futures] :as scheduler} {:keys [id] :as job}]
  (try
    (let [;; add job to backend
          r (backend/add-job! (:backend scheduler) job)]
      ;; remove saved future
      (dosync
       (when (get @futures id)
         (alter futures dissoc id)))
      ;; if job is recurring, add to scheduler again
      (when (:period job)
        (.add-job! scheduler job))
      r)
    (catch Throwable t
      (when-let [handler (-> scheduler :opts :exception-handler)]
        (handler t)))))

(defrecord Scheduler [active? backend futures pool opts]
  IScheduler
  (add-job! [this {:keys [id] :as job}]
    (dosync
     (when-not (get @futures id)
       (let [now (backend/now backend)
             new-start (util/make-start now (:start job) (:period job))
             with-new-start (assoc job :start new-start)]
         (if (<= new-start (+ now 100))
           ;; if new start is in the past or close to now, add immediately
           (add-job-fn this with-new-start)
           ;; otherwise schedule a future to add the job
           (let [future (.schedule @pool
                                   #(add-job-fn this with-new-start)
                                   (- new-start now)
                                   TimeUnit/MILLISECONDS)]
             ;; save scheduled future
             (alter futures assoc id future)
             ;; return updated job
             with-new-start))))))
  (del-job! [_ {:keys [id] :as job}]
    (let [;; cancel and remove saved future
          r1 (dosync
              (when-let [future (get @futures id)]
                (.cancel future false)
                (alter futures dissoc id)
                true))
          ;; remove job from backend
          r2 (backend/del-job! backend job)]
      ;; return truthy if something was removed
      (or r1 r2)))
  (start! [_]
    (when-not @active?
      (reset! active? true)
      (let [num-threads (+ 2 (.availableProcessors (Runtime/getRuntime)))]
        (reset! pool (ScheduledThreadPoolExecutor. num-threads)))
      true))
  (stop! [_]
    (when @active?
      (reset! active? false)
      (.shutdown @pool)
      true)))
