(ns buckshot.util
  (:require [buckshot.backend :as backend]
            [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]
            [clj-time.periodic :as time-periodic]
            [clojure.stacktrace :as st]))

(def period-fns {:year time/years
                 :month time/months
                 :week time/weeks
                 :day time/days
                 :hour time/hours
                 :minute time/minutes
                 :second time/seconds
                 :milli time/millis})

(defn ->period [[x unit]]
  ((get period-fns unit) x))

(defn ->millis [[x unit]]
  (-> [x unit] ->period (.toStandardDuration) (.getMillis)))

(defn make-id []
  (str (java.util.UUID/randomUUID)))

(defn next-start [now start [x unit]]
  (let [n (time-coerce/to-date-time now)
        s (time-coerce/to-date-time start)]
    (->> (time-periodic/periodic-seq s (->period [x unit]))
         (drop-while #(time/before? % n))
         (first))))

(defn make-start [now start period]
  (-> (cond
        ;; no start = use now
        (nil? start)  now
        ;; no period but has start = use start
        (nil? period) start
        ;; has start and period, use both to compute next start
        :else         (next-start now start period))
      (time-coerce/to-long)))

(defn make-job [now fn & [opts]]
  {:args (:args opts)
   :created now
   :fn fn
   :from (.getHostName (java.net.InetAddress/getLocalHost))
   :id (or (:id opts) (make-id))
   :period (:period opts)
   :queue (or (:queue opts) :default)
   :start (make-start now (:start opts) (:period opts))})

(defn log!
  "Publishes a log entry on the :log channel via backend pub-sub mechanism."
  [type worker & [extra]]
  (let [b (:backend worker)
        now (backend/now b)]
    (backend/publish! b :log (merge {:type type
                                     :worker (-> worker :opts :id)
                                     :date (pr-str (java.util.Date. now))
                                     :timestamp now}
                                    extra))))

(defn stack-trace
  "Returns a Throwable's stack trace."
  [t]
  (with-out-str (st/print-stack-trace t)))
