(ns buckshot.middleware
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]))

(defn delete-job
  "Middleware to delete a completed job from the system."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (try
      (handler data)
      (finally
        (backend/del-job! (:backend worker) (:id job) (:queue job))))))

(defn log-start-finish
  "Middleware to log job start and finish."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (let [b (:backend worker)
          start (backend/now b)]
      (try
        (util/log! :start worker {:job job})
        (let [result (handler data)]
          (util/log! :finish worker {:duration (- (backend/now b) start)
                                     :job job})
          result)
        (catch Throwable t
          (util/log! :exception worker {:duration (- (backend/now b) start)
                                        :job job
                                        :trace (util/stack-trace t)})
          (throw t))))))

(defn publish-result
  "Middleware to publish a job result."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (try
      (let [result (handler data)]
        (backend/publish! (:backend worker) (:id job) result))
      (catch Throwable t
        (backend/publish! (:backend worker) (:id job) nil)
        (throw t)))))

(defn rethrow-exception
  "Middleware to re-throw if a job returns an exception."
  [handler]
  (fn [data]
    (let [r (handler data)]
      (if (instance? Throwable r)
        (throw r)
        r))))

(defn warn-unknown-fn
  "Middleware to warn about unknown job functions."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (if-not (get (:fn-map worker) (:fn job))
      (throw (Exception. (str "Unknown job fn: " (:fn job))))
      (handler data))))
