(ns buckshot.scheduler
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]
            [clj-time.coerce :as time-coerce]
            [clojurewerkz.quartzite.conversion :as qc]
            [clojurewerkz.quartzite.jobs :as qj]
            [clojurewerkz.quartzite.schedule.calendar-interval :as qsci]
            [clojurewerkz.quartzite.scheduler :as qs]
            [clojurewerkz.quartzite.triggers :as qt]))

(defprotocol IScheduler
  (add-job! [this fn job-opts])
  (del-job! [this id queue])
  (start!   [this])
  (stop!    [this]))

(qj/defjob AddJob [ctx]
  (let [add-fn (-> ctx qc/from-job-data (get "add-fn"))]
    (add-fn)))

(defn- add-to-backend [{:keys [backend] :as scheduler} fn job-opts]
  (let [now (backend/now backend)
        job (util/make-job now fn job-opts)]
    (backend/add-job! backend job)))

(defn- with-interval [qs {:keys [period] :as job-opts}]
  (if-let [[x unit] period]
    (case unit
      :year   (qsci/with-interval-in-years qs x)
      :month  (qsci/with-interval-in-months qs x)
      :week   (qsci/with-interval-in-weeks qs x)
      :day    (qsci/with-interval-in-days qs x)
      :hour   (qsci/with-interval-in-hours qs x)
      :minute (qsci/with-interval-in-minutes qs x)
      :second (qsci/with-interval-in-seconds qs x))
    qs))

(defrecord Scheduler [active? backend quartz-scheduler opts]
  IScheduler
  (add-job! [this fn {:keys [id] :as job-opts}]
    (when-not (qs/get-job quartz-scheduler (qj/key id))
      (if (and (nil? (:period job-opts))
               (nil? (:start job-opts)))
        ;; add non-recurring jobs starting now immediately (for unit tests)
        (add-to-backend this fn job-opts)
        ;; add everything else thru quartz
        (let [now (backend/now backend)
              qj (qj/build
                  (qj/of-type AddJob)
                  (qj/using-job-data {"add-fn" #(add-to-backend this fn job-opts)})
                  (qj/with-identity (qj/key id)))
              qs (-> (qsci/schedule)
                     (with-interval job-opts))
              qt (qt/build
                  (qt/start-at (-> (:start job-opts)
                                   (or now)
                                   (time-coerce/to-date)))
                  (qt/with-schedule qs))]
          (qs/schedule quartz-scheduler qj qt)))))
  (del-job! [_ id queue]
    (let [;; delete quartz job
          r1 (qs/delete-job quartz-scheduler (qj/key id))
          ;; delete job from backend
          r2 (backend/del-job! backend id queue)]
      ;; return truthy if something was deleted
      (or r1 r2)))
  (start! [_]
    (when-not @active?
      (reset! active? true)
      (qs/pause-all! quartz-scheduler)
      true))
  (stop! [_]
    (when @active?
      (reset! active? false)
      (qs/resume-all! quartz-scheduler)
      true)))
