(ns buckshot.util
  (:require [buckshot.backend :as backend]
            [clj-time.core :as time]
            [clojure.stacktrace :as st]))

(defn ->period [[x unit]]
  (case unit
    :year   (time/years x)
    :month  (time/months x)
    :week   (time/weeks x)
    :day    (time/days x)
    :hour   (time/hours x)
    :minute (time/minutes x)
    :second (time/seconds x)
    :milli  (time/millis x)))

(defn ->millis [[x unit]]
  (-> [x unit] ->period (.toStandardDuration) (.getMillis)))

(defn make-id []
  (str (java.util.UUID/randomUUID)))

(defn make-job [now fn & [opts]]
  {:args (:args opts)
   :created now
   :fn fn
   :from (.getHostName (java.net.InetAddress/getLocalHost))
   :id (or (:id opts) (make-id))
   :period (:period opts)
   :queue (or (:queue opts) :default)
   :start now})

(defn log!
  "Publishes a log entry on the :log channel via backend pub-sub mechanism."
  [type worker & [extra]]
  (let [b (:backend worker)
        now (backend/now b)]
    (backend/publish! b :log (merge {:type type
                                     :worker (-> worker :opts :id)
                                     :date (pr-str (java.util.Date. now))
                                     :timestamp now}
                                    extra))))

(defn stack-trace
  "Returns a Throwable's stack trace."
  [t]
  (with-out-str (st/print-stack-trace t)))
