(ns buckshot.queue
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]
            [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]))

(defprotocol IQueue
  (specs [this])
  (scheduled [this])
  (processing [this])
  (add-specs! [this specs])
  (remove-spec! [this spec])
  (schedule-jobs! [this])
  (unschedule-job! [this job])
  (next-job [this fns])
  (take-job! [this job])
  (finish-job! [this job])
  (publish! [this channel message])
  (subscribe! [this channel f]))

(def period-fns {:year time/years
                 :month time/months
                 :week time/weeks
                 :day time/days
                 :hour time/hours
                 :minute time/minutes
                 :second time/secs})

(defn- upcoming-starts [t period]
  (let [now (util/now-ms)
        p ((get period-fns period) 1)]
    (->> t
         time-coerce/from-long
         (iterate #(time/plus % p))
         (map time-coerce/to-long)
         (drop-while #(< % now)))))

(defn- upcoming-jobs [spec]
  (let [{:keys [start-time period]} spec]
    (for [start (take 3 (upcoming-starts start-time period))]
      (assoc spec :start-time start))))

(defn- ->jobs [specs]
  (let [one-time (remove :period specs)
        recurring (filter :period specs)]
    (concat one-time (mapcat upcoming-jobs recurring))))

(defmacro atomically [& body]
  `(backend/atomically ~'backend
                       (fn [] ~@body)))

(defrecord Queue [backend]
  IQueue
  (specs [_]
    (backend/hall backend :specs))
  (scheduled [_]
    (->> (backend/colls backend)
         (remove #{:specs :processing})
         (mapcat #(backend/zall backend %))))
  (processing [_]
    (backend/zall backend :processing))
  (add-specs! [_ specs]
    (let [now (util/now-ms)]
      (atomically
       (doseq [s specs
               :let [spec (update-in s [:start-time] #(or % now))]]
         (backend/hadd! backend :specs (:id spec) spec)))))
  (remove-spec! [_ {:keys [id fn] :as spec}]
    (atomically
     (backend/hrem! backend :specs id)
     (backend/del! backend fn)))
  (schedule-jobs! [this]
    (doseq [job (-> this specs ->jobs)
            :let [{:keys [id start-time fn]} job]]
      (atomically
       (when (and (backend/hexists? backend :specs id)
                  (not (backend/zexists? backend :processing job)))
         (backend/zadd! backend fn start-time job)))))
  (unschedule-job! [_ {:keys [fn] :as job}]
    (atomically
     (backend/zrem! backend fn job)))
  (next-job [_ fns]
    (let [now (util/now-ms)]
      (some #(backend/zmin backend % now) fns)))
  (take-job! [_ {:keys [start-time fn] :as job}]
    (atomically
     (when (backend/zexists? backend fn job)
       (backend/zrem! backend fn job)
       (backend/zadd! backend :processing start-time job))))
  (finish-job! [_ {:keys [id period] :as job}]
    (atomically
     (when-not period
       (backend/hrem! backend :specs id))
     (backend/zrem! backend :processing job)))
  (publish! [_ channel message]
    (backend/publish! backend channel message))
  (subscribe! [_ channel f]
    (backend/subscribe! backend channel f)))

(defn make [params]
  (Queue. (:backend params)))
