(ns buckshot.queue
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]
            [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]))

(defprotocol IQueue
  (specs [this])
  (scheduled [this])
  (processing [this])
  (add-specs! [this specs])
  (remove-spec! [this spec])
  (schedule-jobs! [this])
  (unschedule-job! [this job])
  (next-job [this fns])
  (take-job! [this job])
  (finish-job! [this job])
  (publish! [this channel message])
  (subscribe! [this channel f]))

(def period-fns {:year time/years
                 :month time/months
                 :week time/weeks
                 :day time/days
                 :hour time/hours
                 :minute time/minutes
                 :second time/secs})

(defn- upcoming-starts [t period]
  (let [now (util/now-ms)
        p ((get period-fns period) 1)]
    (->> t
         time-coerce/from-long
         (iterate #(time/plus % p))
         (map time-coerce/to-long)
         (drop-while #(< % now)))))

(defn- upcoming-jobs [spec]
  (let [{:keys [start-time period]} spec]
    (for [start (take 3 (upcoming-starts start-time period))]
      (assoc spec :start-time start))))

(defmacro atomically [colls & body]
  `(backend/atomically ~'backend
                       ~colls
                       (fn [] ~@body)))

(defrecord Queue [backend]
  IQueue
  (specs [_]
    (concat (backend/hall backend :one-time)
            (backend/hall backend :recurring)))
  (scheduled [_]
    (->> (backend/colls backend)
         (remove #{:one-time :recurring :processing})
         (mapcat #(backend/zall backend %))))
  (processing [_]
    (backend/zall backend :processing))
  (add-specs! [_ specs]
    (doseq [{:keys [id fn period] :as s} specs
            :let [spec (update-in s [:start-time] #(or % (util/now-ms)))
                  coll (if period :recurring :one-time)]]
      (atomically
       []
       (when-not (backend/hexists? backend coll id)
         (backend/hadd! backend coll id spec)
         (backend/zadd! backend fn (:start-time spec) spec)))))
  (remove-spec! [_ {:keys [id fn period] :as spec}]
    (let [coll (if period :recurring :one-time)]
      (atomically
       []
       (backend/hrem! backend coll id)
       (backend/del! backend fn))))
  (schedule-jobs! [_]
    (doseq [job (->> (backend/hall backend :recurring)
                     (mapcat upcoming-jobs))
            :let [{:keys [start-time fn]} job]]
      (atomically
       []
       (when-not (backend/zexists? backend fn job)
         (backend/zadd! backend fn start-time job)))))
  (unschedule-job! [_ {:keys [fn] :as job}]
    (atomically
     []
     (backend/zrem! backend fn job)))
  (next-job [_ fns]
    (let [now (util/now-ms)]
      (some #(backend/zmin backend % now) fns)))
  (take-job! [_ {:keys [start-time fn] :as job}]
    (atomically
     [fn]
     (when (backend/zexists? backend fn job)
       (backend/zrem! backend fn job)
       (backend/zadd! backend :processing start-time job))))
  (finish-job! [_ {:keys [id period] :as job}]
    (atomically
     []
     (when-not period
       (backend/hrem! backend :one-time id))
     (backend/zrem! backend :processing job)))
  (publish! [_ channel message]
    (backend/publish! backend channel message))
  (subscribe! [_ channel f]
    (backend/subscribe! backend channel f)))

(defn make [params]
  (Queue. (:backend params)))
