(ns buckshot.queue
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]
            [clj-time.coerce :as time-coerce]
            [clj-time.core :as time]))

(defprotocol IQueue
  (add-job! [this job])
  (remove-job! [this job-id])
  (next-job [this fns])
  (take-job! [this job-id])
  (finish-job! [this job-id])
  (publish! [this channel message])
  (subscribe! [this channel f]))

(def time-unit-fns {:year time/years
                    :month time/months
                    :week time/weeks
                    :day time/days
                    :hour time/hours
                    :minute time/minutes
                    :second time/secs})

(defn- upcoming-start [start period]
  (let [now (util/now-ms)
        [n unit] period
        p ((time-unit-fns unit) n)]
    (->> (time-coerce/from-long start)
         (iterate #(time/plus % p))
         (map time-coerce/to-long)
         (drop-while #(< % now))
         first)))

(defn- get-start [job]
  (let [{:keys [start period]} job]
    (if (nil? period)
      (or start (util/now-ms))
      (upcoming-start (or start (util/today-ms)) period))))

(defmacro atomically [colls & body]
  `(backend/atomically ~'backend
                       ~colls
                       (fn [] ~@body)))

(defrecord Queue [backend]
  IQueue
  (add-job! [_ {:keys [id fn] :as job}]
    (atomically
     []
     (when-not (backend/hget backend :jobs id)
       (let [s (get-start job)]
         (backend/hadd! backend :jobs id (assoc job :start s))
         (backend/zadd! backend fn s id)))))
  (remove-job! [_ {:keys [id fn] :as job}]
    (atomically
     []
     (backend/hrem! backend :jobs id)
     (backend/zrem! backend fn id)))
  (next-job [_ fns]
    (let [now (util/now-ms)]
      (->> (shuffle fns)
           (some #(backend/zmin backend % now))
           (backend/hget backend :jobs))))
  (take-job! [_ {:keys [id fn start] :as job}]
    (atomically
     [fn]
     (when (backend/zexists? backend fn id)
       (backend/zrem! backend fn id)
       (backend/zadd! backend :processing start id))))
  (finish-job! [_ {:keys [id] :as job}]
    (atomically
     []
     (backend/hrem! backend :jobs id)
     (backend/zrem! backend :processing id)))
  (publish! [_ channel message]
    (backend/publish! backend channel message))
  (subscribe! [_ channel f]
    (backend/subscribe! backend channel f)))

(defn make [params]
  (Queue. (:backend params)))
