(ns buckshot.worker
  (:require [buckshot.core :as buckshot]
            [buckshot.queue :as queue]
            [buckshot.util :as util]
            [clojure.stacktrace :as st]))

(defprotocol IWorker
  (start! [this])
  (stop! [this]))

(defn- stack-trace [t]
  (with-out-str (st/print-stack-trace t)))

(defn- log! [{:keys [queue] :as worker} type job extra]
  (buckshot/log! queue (merge {:type type
                               :worker (:id worker)
                               :job job}
                              extra)))

(defn- handle-job! [{:keys [queue] :as worker} job]
  (log! worker :start job nil)
  (let [f (get (:fns worker) (:fn job))
        start (util/now-ms)
        result (apply f (:args job))
        end (util/now-ms)]
    (log! worker :finish job {:duration (- end start)})
    (queue/finish-job! queue job)
    (queue/publish! queue (:id job) result)))

(defn- handle-exception! [{:keys [error-handler queue] :as worker} t job]
  (log! worker :exception job {:trace (stack-trace t)})
  (queue/finish-job! queue job)
  (when error-handler
    (try (error-handler t)
         (catch Throwable t2
           (log! worker :exception job {:trace (stack-trace t2)})))))

(defrecord Worker [id error-handler fns queue sleep active?]
  IWorker
  (start! [this]
    (when-not @active?
      (reset! active? true)
      (future
        (while @active?
          (if-let [j (try (queue/next-job queue (keys fns))
                          (catch Throwable t
                            (handle-exception! this t nil)
                            nil))]
            (try (if (queue/take-job! queue j)
                   (handle-job! this j)
                   (Thread/sleep (rand-int 1000)))
                 (catch Throwable t
                   (handle-exception! this t j)))
            (Thread/sleep sleep))))))
  (stop! [_]
    (reset! active? false)))

(defn make [{:keys [id error-handler fns queue sleep] :as params}]
  (let [w (Worker. id error-handler fns queue (or sleep 500) (atom false))]
    (start! w)
    w))
