(ns buckshot.worker
  (:require [buckshot.core :as buckshot]
            [buckshot.queue :as queue]
            [buckshot.util :as util]
            [clojure.stacktrace :as st]))

(defprotocol IWorker
  (start! [this])
  (stop! [this]))

(defn- stack-trace [t]
  (with-out-str (st/print-stack-trace t)))

(defn- log! [{:keys [queue] :as worker} type job extra]
  (buckshot/log! queue (merge {:type type
                               :worker (:id worker)
                               :job job}
                              extra)))

(defn- handle-exception! [t {:keys [queue] :as worker} job]
  (log! worker :exception job {:trace (stack-trace t)})
  (queue/publish! queue (:id job) nil)
  (queue/finish-job! queue job)
  (throw t))

(defn- handle-job! [{:keys [queue] :as worker} job]
  (log! worker :start job nil)
  (let [f (get (:fns worker) (:fn job))
        start (util/now-ms)
        result (try (apply f (:args job))
                    (catch Throwable t
                      (handle-exception! t worker job)))
        end (util/now-ms)]
    (log! worker :finish job {:duration (- end start)})
    (queue/publish! queue (:id job) result)
    (queue/finish-job! queue job)))

(defrecord Worker [id error-handler fns queue sleep active?]
  IWorker
  (start! [this]
    (when-not @active?
      (reset! active? true)
      (future (while @active?
                (try (if-let [j (queue/next-job queue (keys fns))]
                       (if (queue/take-job! queue j)
                         (handle-job! this j)
                         (Thread/sleep (rand-int 1000)))
                       (Thread/sleep sleep))
                     (catch Throwable t
                       (when error-handler
                         (error-handler t))))))))
  (stop! [_]
    (reset! active? false)))

(defn make [{:keys [id error-handler fns queue sleep] :as params}]
  (let [w (Worker. id error-handler fns queue (or sleep 500) (atom false))]
    (start! w)
    w))
