(ns buckshot.backend.memory
  (:require [buckshot.backend :as backend]))

(defn- next-by-queue [queue jobs now]
  (->> jobs
       (sort-by :start)
       (filter #(-> % :queue (= queue)))
       (filter #(-> % :start (<= now)))
       (first)))

(defrecord MemoryBackend [jobs subscribers]
  backend/IBackend
  (add-job! [_ {:keys [id] :as job}]
    (dosync
     (when-not (get @jobs id)
       (let [with-status (assoc job :status :enqueued)]
         (alter jobs assoc id with-status)
         with-status))))
  (del-job! [_ {:keys [id] :as job}]
    (dosync
     (when (get @jobs id)
       (alter jobs dissoc id))))
  (get-jobs [_ ids]
    (vec (map @jobs ids)))
  (next-job [this queues]
    (let [now (backend/now this)
          js (vals @jobs)]
      (some #(next-by-queue % js now) queues)))
  (claim-job! [_ {:keys [id] :as job}]
    (dosync
     (when (-> @jobs (get id) :status (= :enqueued))
       (alter jobs assoc-in [id :status] :processing)
       job)))
  (queues [_]
    (let [js (vals @jobs)]
      (->> js
           (filter #(-> % :status (= :enqueued)))
           (map :queue)
           (distinct))))
  (enqueued-ids [_ queues]
    (let [js (vals @jobs)]
      (into {} (for [q queues]
                 [q (->> js
                         (filter #(-> % :status (= :enqueued)))
                         (filter #(-> % :queue (= q)))
                         (map :id))]))))
  (processing-ids [_]
    (let [js (vals @jobs)]
      (->> js
           (filter #(-> % :status (= :processing)))
           (map :id))))
  (publish! [_ channel message]
    (doseq [f (get @subscribers channel)]
      (f message)))
  (subscribe! [_ channel f]
    (swap! subscribers update-in [channel] conj f))
  (now [_]
    (System/currentTimeMillis)))

(defn make []
  (->MemoryBackend (ref {}) (atom {})))
