(ns buckshot.core
  (:require [buckshot.backend :as backend]
            [buckshot.backend.memory :as mem-backend]
            [buckshot.backend.redis :as redis-backend]
            [buckshot.middleware :as mid]
            [buckshot.scheduler :as scheduler]
            [buckshot.util :as util]
            [buckshot.worker :as worker]))

(defn enqueue!
  "Enqueues a job, returns enqueued job."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [j (util/make-job (backend/now backend) fn opts)]
    (scheduler/add-job! scheduler j)))

(defn enqueue-p!
  "Enqueues a job, returns enqueued job and a promise for the result."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [id (or (:id opts) (util/make-id))
        p (promise)]
    (backend/subscribe! backend id #(deliver p %))
    (list (enqueue! scheduler fn (assoc opts :id id)) p)))

(defn make-backend
  "Returns new Buckshot backend of specified type."
  [type & [opts]]
  (case type
    :memory (mem-backend/make)
    :redis  (redis-backend/make opts)))

(defn make-scheduler
  "Returns new Buckshot scheduler."
  [backend & [opts]]
  (let [{:keys [exception-handler id sleep]} opts
        s (scheduler/->Scheduler (atom false)
                                 backend
                                 (ref {})
                                 {:exception-handler exception-handler
                                  :id (or id (util/make-id))
                                  :sleep (if sleep
                                           (util/->millis sleep)
                                           1000)})]
    (scheduler/start! s)
    s))

(defn make-worker
  "Returns new Buckshot worker."
  [backend fn-map & [opts]]
  (let [{:keys [exception-handler id middleware queues sleep]} opts
        w (worker/->Worker (atom false)
                           backend
                           fn-map
                           {:exception-handler exception-handler
                            :id (or id (util/make-id))
                            :middleware (concat [mid/handle-unknown-fn
                                                 mid/rethrow-exception
                                                 mid/log-start-finish]
                                                middleware
                                                [mid/handle-exception])
                            :queues (or queues [:default])
                            :sleep (if sleep
                                     (util/->millis sleep)
                                     1000)})]
    (worker/start! w)
    w))
