(ns buckshot.scheduler
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]))

(defprotocol IScheduler
  (add-job! [this job])
  (del-job! [this job])
  (start!   [this])
  (stop!    [this]))

(defrecord Scheduler [active? backend jobs opts]
  IScheduler
  (add-job! [_ {:keys [id] :as job}]
    (dosync
     (when-not (get @jobs id)
       (when (:period job)
         (alter jobs assoc id job))
       (backend/add-job! backend job))))
  (del-job! [_ {:keys [id] :as job}]
    (dosync
     (alter jobs dissoc id))
    (backend/del-job! backend job))
  (start! [_]
    (when-not @active?
      (reset! active? true)
      (future (while @active?
                (try (doseq [[_ j] @jobs
                             :let [now (backend/now backend)
                                   new-j (util/make-job now (:fn j) j)]]
                       (backend/add-job! backend new-j))
                     (Thread/sleep (:sleep opts))
                     (catch Throwable t
                       (when-let [handler (:exception-handler opts)]
                         (handler t)))))))
    true)
  (stop! [_]
    (reset! active? false)
    true))
