(ns buckshot.middleware
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]))

(defn- log!
  "Publishes a log entry on the :log channel via backend pub-sub mechanism."
  [type worker job & [extra]]
  (let [b (:backend worker)
        now (backend/now b)]
    (backend/publish! b :log (merge {:type type
                                     :worker (-> worker :opts :id)
                                     :job job
                                     :date (pr-str (java.util.Date. now))
                                     :timestamp now}
                                    extra))))

(defn handle-exception
  "Middleware to handle exceptions - logs the exception, publishes nil as the result of the job, and removes the job from the system."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (try (handler data)
         (catch Throwable t
           (let [b (:backend worker)]
             (log! :exception worker job {:trace (util/stack-trace t)})
             (backend/publish! b (:id job) nil)
             (backend/del-job! b job)
             (throw t))))))

(defn handle-unknown-fn
  "Middleware to throw an exception if the job fn is unknown."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (if-not (get (:fn-map worker) (:fn job))
      (throw (Exception. (str "Unknown job fn: " (:fn job))))
      (handler data))))

(defn log-start-finish
  "Middleware to log job start and finish."
  [handler]
  (fn [{:keys [job worker] :as data}]
    (log! :start worker job)
    (let [b (:backend worker)
          start (backend/now b)
          r (handler data)
          end (backend/now b)]
      (log! :finish worker job {:duration (- end start)})
      r)))

(defn rethrow-exception
  "Middleware to re-throw if a job returns an exception."
  [handler]
  (fn [data]
    (let [r (handler data)]
      (if (instance? Throwable r)
        (throw r)
        r))))
