(ns buckshot.core
  (:require [buckshot.backend :as backend]
            [buckshot.backend.memory :as mem-backend]
            [buckshot.backend.redis :as redis-backend]
            [buckshot.middleware :as mid]
            [buckshot.scheduler :as scheduler]
            [buckshot.util :as util]
            [buckshot.worker :as worker]))

(defn enqueue!
  "Enqueues a job, returns enqueued job.

  'fn' should be a key in worker function maps.

  'opts' may contain the following:

  :args - seq of arguments for the job function.
  :id - ID for the job. If a job with the same ID is already in the
        system, the new job will not be enqueued. Default is a random UUID.
  :period - if specified, the job will recur with the supplied period, i.e.
            [1 :hour]. See buckshot.util/period-fns for supported time units.
  :queue - queue for the job. Default is :default.
  :start - start time for the job. Must be a Joda DateTime to correctly handle
           Daylight Savings Time. Default is 'now' in UTC."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [id (or (:id opts) (util/make-id))]
    (scheduler/add-job! scheduler fn (assoc opts :id id))))

(defn enqueue-p!
  "Enqueues a job, returns enqueued job and a promise for the result. See
  'enqueue!' for supported options."
  [{:keys [backend] :as scheduler} fn & [opts]]
  (let [id (or (:id opts) (util/make-id))
        p (promise)]
    (backend/subscribe! backend id #(deliver p %))
    [(enqueue! scheduler fn (assoc opts :id id)) p]))

(defn make-backend
  "Returns new Buckshot backend of specified type. See each type's (make)
  function for supported options."
  [type & [opts]]
  (case type
    :memory (mem-backend/make)
    :redis  (redis-backend/make opts)))

(defn make-scheduler
  "Returns new Buckshot scheduler.

  'opts' may contain the following:

  :exception-handler - a function that will be called with any exception thrown
                       by the scheduler.
  :id - ID for the scheduler. Default is a random UUID.
  :sleep - if specified, the scheduler will sleep for the supplied period, i.e.
           [1 :hour], after scheduling jobs. See buckshot.util/period-fns for
           supported time units (all except :year and :month, which can have
           variable length). Default is 1 second."
  [backend & [opts]]
  (let [{:keys [exception-handler id sleep]} opts
        s (scheduler/->Scheduler (atom false)
                                 backend
                                 (ref {})
                                 {:exception-handler exception-handler
                                  :id (or id (util/make-id))
                                  :sleep (if sleep
                                           (util/->millis sleep)
                                           1000)})]
    (scheduler/start! s)
    s))

(defn make-worker
  "Returns new Buckshot worker.

  'fn-map' should be a map of keys to job functions, which may have arbitrary
  arity. See buckshot.test.core for examples.

  'opts' may contain the following:

  :exception-handler - a function that will be called with any exception thrown
                       by the scheduler.
  :id - ID for the worker. Default is a random UUID.
  :middleware - seq of worker middleware functions (these are just like Ring
                middleware). Middleware are run in nested order; the last one
                starts first and finishes last. See buckshot.middleware and
                buckshot.test.core for examples.
  :queues - seq of job queues to process. Queues are prioritized by order, with
            the first queue having the highest priority. Default is [:default].
  :sleep - if specified, the worker will sleep for the supplied period, i.e.
           [1 :hour], after processing jobs. See buckshot.util/period-fns for
           supported time units (all except :year and :month, which can have
           variable length). Default is 1 second."
  [backend fn-map & [opts]]
  (let [{:keys [exception-handler id middleware queues sleep]} opts
        w (worker/->Worker (atom false)
                           backend
                           fn-map
                           {:exception-handler exception-handler
                            :id (or id (util/make-id))
                            :middleware (concat [mid/warn-unknown-fn
                                                 mid/rethrow-exception
                                                 mid/publish-result
                                                 mid/delete-job]
                                                middleware
                                                [mid/log-start-finish])
                            :queues (or queues [:default])
                            :sleep (if sleep
                                     (util/->millis sleep)
                                     1000)})]
    (worker/start! w)
    w))
