(ns buckshot.scheduler
  (:require [buckshot.backend :as backend]
            [buckshot.util :as util]))

(defprotocol IScheduler
  (add-job! [this fn job-opts])
  (del-job! [this job])
  (start!   [this])
  (stop!    [this]))

(defrecord Scheduler [active? backend jobs opts]
  IScheduler
  (add-job! [_ fn {:keys [id] :as job-opts}]
    (dosync
     (when-not (get @jobs id)
       (when (:period job-opts)
         (alter jobs assoc id [fn job-opts]))
       (let [now (backend/now backend)
             j (util/make-job now fn job-opts)]
         (backend/add-job! backend j)))))
  (del-job! [_ {:keys [id] :as job}]
    (dosync
     (alter jobs dissoc id))
    (backend/del-job! backend job))
  (start! [_]
    (when-not @active?
      (reset! active? true)
      (future (while @active?
                (try (doseq [[_ [fn job-opts]] @jobs
                             :let [now (backend/now backend)
                                   j (util/make-job now fn job-opts)]]
                       (backend/add-job! backend j))
                     (Thread/sleep (:sleep opts))
                     (catch Throwable t
                       (when-let [handler (:exception-handler opts)]
                         (handler t)))))))
    true)
  (stop! [_]
    (reset! active? false)
    true))
